<?php

namespace App\Models;

use App\Models\User;
use App\Models\Branch;
use App\Models\Supplier;
use App\Models\SupplierLedger;
use App\Models\PaymentAllocation;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;

class SupplierPayment extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'payment_number',
        'supplier_id',
        'branch_id',
        'user_id',
        'payment_date',
        'amount',
        'payment_method',
        'reference_number',
        'notes',
    ];

    protected $casts = [
        'payment_date' => 'date',
        'amount'       => 'integer',
    ];

    /** ═══════════════════════════════════════════════════════
     *  العلاقات
     * ═══════════════════════════════════════════════════════ */

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function allocations(): HasMany
    {
        return $this->hasMany(PaymentAllocation::class);
    }

    public function ledgerEntries(): MorphMany
    {
        return $this->morphMany(SupplierLedger::class, 'ledgerable');
    }

    /** ═══════════════════════════════════════════════════════
     *  السكوبات
     * ═══════════════════════════════════════════════════════ */

    public function scopeForSupplier($query, int $supplierId)
    {
        return $query->where('supplier_id', $supplierId);
    }

    public function scopeForBranch($query, int $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    public function scopeByMethod($query, string $method)
    {
        return $query->where('payment_method', $method);
    }

    public function scopeBetweenDates($query, $from, $to)
    {
        return $query->whereBetween('payment_date', [$from, $to]);
    }

    /** ═══════════════════════════════════════════════════════
     *  الميثودات المساعدة
     * ═══════════════════════════════════════════════════════ */

    // إجمالي المبلغ المخصص على الفواتير
    public function getTotalAllocatedAttribute(): int
    {
        return (int) $this->allocations()->sum('allocated_amount');
    }

    // المبلغ غير المخصص (المتبقي)
    public function getUnallocatedAmountAttribute(): int
    {
        return $this->amount - $this->total_allocated;
    }

    // هل تم تخصيص الدفعة بالكامل؟
    public function isFullyAllocated(): bool
    {
        return $this->unallocated_amount <= 0;
    }
}
