<?php

namespace App\Models;

use App\Models\Branch;
use App\Models\Supplier;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SupplierLedger extends Model
{
    protected $table = 'supplier_ledger';

    protected $fillable = [
        'supplier_id',
        'branch_id',
        'ledgerable_type',
        'ledgerable_id',
        'transaction_date',
        'type',
        'amount',
        'reference',
        'description',
        'notes',
    ];

    protected $casts = [
        'transaction_date' => 'date',
        'amount'           => 'integer',
    ];

    /** ═══════════════════════════════════════════════════════
     *  العلاقات
     * ═══════════════════════════════════════════════════════ */

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function ledgerable()
    {
        return $this->morphTo();
    }

    /** ═══════════════════════════════════════════════════════
     *  السكوبات
     * ═══════════════════════════════════════════════════════ */

    public function scopeForSupplier($query, int $supplierId)
    {
        return $query->where('supplier_id', $supplierId);
    }

    public function scopeForBranch($query, int $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    public function scopeByType($query, string $type)
    {
        return $query->where('type', $type);
    }

    public function scopeDebits($query)
    {
        return $query->where('amount', '>', 0);
    }

    public function scopeCredits($query)
    {
        return $query->where('amount', '<', 0);
    }

    public function scopeBetweenDates($query, $from, $to)
    {
        return $query->whereBetween('transaction_date', [$from, $to]);
    }

    /** ═══════════════════════════════════════════════════════
     *  الميثودات المساعدة
     * ═══════════════════════════════════════════════════════ */

    public function isDebit(): bool
    {
        return $this->amount > 0;
    }

    public function isCredit(): bool
    {
        return $this->amount < 0;
    }
}
