<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

/**
 * إعدادات المتجر الإلكتروني
 */
class StoreSetting extends Model implements HasMedia
{
    use HasFactory, InteractsWithMedia;

    protected $table = 'store_settings';

    /** Media Collections */
    public const MEDIA_LOGO = 'store_logo';

    public const MEDIA_FAVICON = 'store_favicon';

    public const MEDIA_BANNER = 'store_banner';

    protected $fillable = [
        'store_name',
        'store_slug',
        'description',
        'email',
        'phone',
        'address',
        'default_currency_id',
        'tax_rate',
        'prices_include_tax',
        'enable_shipping',
        'free_shipping_threshold',
        'default_shipping_fee',
        'payment_methods',
        'enable_cod',
        'cod_fee',
        'track_inventory',
        'allow_backorders',
        'low_stock_threshold',
        'min_order_amount',
        'max_order_items',
        'require_phone',
        'require_address',
        'enable_coupons',
        'enable_promotions',
        'allow_multiple_coupons',
        'notification_settings',
        'theme_primary',
        'theme_palette',
        'socials',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'advanced_settings',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'tax_rate' => 'decimal:2',
            'prices_include_tax' => 'boolean',
            'enable_shipping' => 'boolean',
            'free_shipping_threshold' => 'integer',
            'default_shipping_fee' => 'integer',
            'payment_methods' => 'array',
            'enable_cod' => 'boolean',
            'cod_fee' => 'integer',
            'track_inventory' => 'boolean',
            'allow_backorders' => 'boolean',
            'low_stock_threshold' => 'integer',
            'min_order_amount' => 'integer',
            'max_order_items' => 'integer',
            'require_phone' => 'boolean',
            'require_address' => 'boolean',
            'enable_coupons' => 'boolean',
            'enable_promotions' => 'boolean',
            'allow_multiple_coupons' => 'boolean',
            'notification_settings' => 'array',
            'theme_palette' => 'array',
            'socials' => 'array',
            'meta_keywords' => 'array',
            'advanced_settings' => 'array',
            'is_active' => 'boolean',
        ];
    }

    protected $appends = [
        'logo_url',
        'favicon_url',
        'banner_url',
    ];

    /* ================== العلاقات ================== */

    public function defaultCurrency(): BelongsTo
    {
        return $this->belongsTo(Currency::class, 'default_currency_id');
    }

    /* ================== Media Library ================== */

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection(self::MEDIA_LOGO)
            ->singleFile()
            ->useDisk('upload')
            ->acceptsMimeTypes(['image/png', 'image/jpeg', 'image/webp', 'image/svg+xml']);

        $this->addMediaCollection(self::MEDIA_FAVICON)
            ->singleFile()
            ->useDisk('upload')
            ->acceptsMimeTypes(['image/png', 'image/x-icon', 'image/vnd.microsoft.icon', 'image/svg+xml']);

        $this->addMediaCollection(self::MEDIA_BANNER)
            ->singleFile()
            ->useDisk('upload')
            ->acceptsMimeTypes(['image/png', 'image/jpeg', 'image/webp']);
    }

    public function registerMediaConversions(?Media $media = null): void
    {
        if (! $media) {
            return;
        }

        if (in_array($media->mime_type, ['image/svg+xml', 'image/x-icon', 'image/vnd.microsoft.icon'], true)) {
            return;
        }

        $this->addMediaConversion('thumb')
            ->width(200)->height(200)
            ->nonQueued();

        $this->addMediaConversion('preview')
            ->width(600)->height(600)
            ->nonQueued();
    }

    /* ================== Accessors ================== */

    public function getLogoUrlAttribute(): ?string
    {
        $url = $this->getFirstMediaUrl(self::MEDIA_LOGO);

        return $url !== '' ? $url : null;
    }

    public function getFaviconUrlAttribute(): ?string
    {
        $url = $this->getFirstMediaUrl(self::MEDIA_FAVICON);

        return $url !== '' ? $url : null;
    }

    public function getBannerUrlAttribute(): ?string
    {
        $url = $this->getFirstMediaUrl(self::MEDIA_BANNER);

        return $url !== '' ? $url : null;
    }

    /* ================== Static Methods ================== */

    /**
     * الحصول على إعدادات المتجر الحالية
     */
    public static function current(): ?self
    {
        return cache()->remember('store_settings', 3600, function () {
            return static::query()->where('is_active', true)->first();
        });
    }

    /**
     * مسح الكاش عند التحديث
     */
    protected static function booted(): void
    {
        static::saved(function () {
            cache()->forget('store_settings');
        });

        static::deleted(function () {
            cache()->forget('store_settings');
        });
    }

    /* ================== Helper Methods ================== */

    /**
     * هل الشحن مجاني؟
     */
    public function isFreeShipping(int $orderTotal): bool
    {
        if (! $this->enable_shipping) {
            return true;
        }

        if ($this->free_shipping_threshold && $orderTotal >= $this->free_shipping_threshold) {
            return true;
        }

        return false;
    }

    /**
     * حساب رسوم الشحن
     */
    public function calculateShippingFee(int $orderTotal): int
    {
        if ($this->isFreeShipping($orderTotal)) {
            return 0;
        }

        return $this->default_shipping_fee ?? 0;
    }

    /**
     * حساب الضريبة
     */
    public function calculateTax(int $amount): int
    {
        if ($this->tax_rate <= 0) {
            return 0;
        }

        return (int) round($amount * ($this->tax_rate / 100));
    }

    /**
     * هل طريقة الدفع متاحة؟
     */
    public function isPaymentMethodEnabled(string $method): bool
    {
        $methods = $this->payment_methods ?? [];

        return in_array($method, $methods, true);
    }
}
