<?php

namespace App\Models;

use App\Enums\MovementDirection;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * حركة مخزنية (وارد / صادر / تسوية).
 * تؤثر مباشرة على رصيد StockBalance بعد الإنشاء.
 *
 * يمكن ربطها بأي مصدر (فاتورة شراء، فاتورة مبيعات، تحويل مخزني، ... إلخ)
 * عبر الحقول polymorphic: source_type + source_id.
 */
class StockMovement extends Model
{
    protected $table = 'stock_movements';

    protected $fillable = [
        'warehouse_id',
        'product_variant_id',
        'direction',          // IN / OUT / ADJ (Enum MovementDirection)
        'source_type',        // اسم الكلاس الكامل للمصدر App\Models\...
        'source_id',          // رقم السجل في جدول المصدر
        'quantity',           // كمية الحركة (INT)
        'unit_cost',          // تكلفة الوحدة (INT - أصغر وحدة عملة)
        'total_cost',         // تكلفة إجمالية (INT)
        'moved_at',
        'reference',          // رقم مرجع نصي إن أحببت (مثلاً رقم فاتورة خارجي)
        'note',
        'user_id',
    ];

    protected $casts = [
        'direction'  => MovementDirection::class,
        'quantity'   => 'int',
        'unit_cost'  => 'int',
        'total_cost' => 'int',
        'moved_at'   => 'datetime',
    ];

    /* ========================= Boot Logic ========================= */

    protected static function booted(): void
    {
        // قبل الإنشاء: حساب total_cost وتثبيت التاريخ
        static::creating(function (self $m) {
            if (is_null($m->total_cost) && ! is_null($m->unit_cost)) {
                $m->total_cost = (int) $m->unit_cost * (int) $m->quantity;
            }

            if (is_null($m->moved_at)) {
                $m->moved_at = now();
            }
        });

        // بعد الإنشاء: تحديث رصيد المستودع داخل Transaction + lockForUpdate
        static::created(function (self $m) {
            DB::transaction(function () use ($m) {
                /** @var \App\Models\StockBalance|null $balance */
                $balance = StockBalance::query()
                    ->where('warehouse_id', $m->warehouse_id)
                    ->where('product_variant_id', $m->product_variant_id)
                    ->lockForUpdate()
                    ->first();

                if (! $balance) {
                    $balance = new StockBalance([
                        'warehouse_id'       => $m->warehouse_id,
                        'product_variant_id' => $m->product_variant_id,
                        'on_hand'            => 0,
                        'reserved_qty'       => 0,
                        'min_stock_level'    => 0,
                        'max_stock_level'    => 0,
                        'reorder_point'      => 0,
                        'reorder_quantity'   => 0,
                        'last_unit_cost'     => null,
                        'last_movement_at'   => null,
                    ]);
                }

                $sign  = $m->direction->sign();          // +1 / -1 / حسب الـ Enum
                $delta = $sign * (int) $m->quantity;

                // الرصيد الجديد قبل الحماية
                $newOnHand = (int) $balance->on_hand + $delta;

                // منع الرصيد السلبي كطبقة حماية أخيرة
                if ($newOnHand < 0) {
                    // يمكن هنا رمي استثناء بدل القص إلى 0 حسب سياسة النظام
                    // throw new \RuntimeException('لا يمكن أن يصبح رصيد المخزون سالباً.');
                    $newOnHand = 0;
                }

                $balance->on_hand          = $newOnHand;
                $balance->last_movement_at = $m->moved_at;

                // عند الوارد، حدّث آخر تكلفة معلومة
                if (! is_null($m->unit_cost) && $m->direction === MovementDirection::IN) {
                    $balance->last_unit_cost = (int) $m->unit_cost;
                }

                $balance->save();
            });
        });
    }

    /* ========================= العلاقات ========================= */

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function variant(): BelongsTo
    {
        return $this->belongsTo(ProductVariant::class, 'product_variant_id');
    }

    /**
     * المصدر البوليمورفي للحركة (Sales, PurchaseInvoice, ClientReceipt, ...).
     */
    public function source(): MorphTo
    {
        return $this->morphTo();
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /* ========================= Scopes أساسية ========================= */

    public function scopeForVariant($q, int $variantId)
    {
        return $q->where('product_variant_id', $variantId);
    }

    public function scopeForWarehouse($q, int $warehouseId)
    {
        return $q->where('warehouse_id', $warehouseId);
    }

    public function scopeBetween($q, $from, $to)
    {
        return $q->whereBetween('moved_at', [$from, $to]);
    }

    /**
     * حركات تخص فرع معيّن (عبر المستودعات المرتبطة بالفرع).
     */
    public function scopeForBranch($q, int $branchId)
    {
        return $q->whereHas('warehouse', function ($w) use ($branchId) {
            $w->whereHas('branches', function ($b) use ($branchId) {
                $b->where('branches.id', $branchId);
            });
        });
    }

    /* ========================= Scopes للتقارير ========================= */

    /**
     * حركات لمنتج معيّن (كل نسخه).
     */
    public function scopeForProduct($q, int $productId)
    {
        return $q->whereHas('variant', function ($qq) use ($productId) {
            $qq->where('product_id', $productId);
        });
    }

    /**
     * حركات واردة فقط.
     */
    public function scopeIn($q)
    {
        return $q->where('direction', MovementDirection::IN);
    }

    /**
     * حركات صادرة فقط.
     */
    public function scopeOut($q)
    {
        return $q->where('direction', MovementDirection::OUT);
    }

    /**
     * حركات تسوية فقط.
     */
    public function scopeAdjust($q)
    {
        return $q->where('direction', MovementDirection::ADJ);
    }

    /**
     * حركات لمصدر معيّن (حسب نوع الموديل الكامل).
     *
     * مثال:
     * StockMovement::forSource(App\Models\Sales::class)->get();
     */
    public function scopeForSource($q, string $sourceType)
    {
        return $q->where('source_type', $sourceType);
    }

    /**
     * حركات ليوم معيّن (حسب moved_at).
     */
    public function scopeForDate($q, $date)
    {
        return $q->whereDate('moved_at', $date);
    }

    /**
     * حركات لشهر معيّن (yyyy-mm).
     */
    public function scopeForMonth($q, int $year, int $month)
    {
        return $q
            ->whereYear('moved_at', $year)
            ->whereMonth('moved_at', $month);
    }
}
