<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * رصيد المخزون لكل (مستودع × متغير منتج).
 * يحتوي الرصيد الفعلي + المحجوز + حدود المخزون + آخر تكلفة + وقت آخر حركة.
 */
class StockBalance extends Model
{
    protected $table = 'stock_balances';

    protected $fillable = [
        'warehouse_id',
        'product_variant_id',
        'on_hand',
        'reserved_qty',
        'min_stock_level',
        'max_stock_level',
        'reorder_point',
        'reorder_quantity',
        'last_unit_cost',
        'last_movement_at',
    ];

    protected $casts = [
        'on_hand'          => 'int',
        'reserved_qty'     => 'int',
        'min_stock_level'  => 'int',
        'max_stock_level'  => 'int',
        'reorder_point'    => 'int',
        'reorder_quantity' => 'int',
        'last_unit_cost'   => 'int',
        'last_movement_at' => 'datetime',
    ];

    /* ========================= العلاقات ========================= */

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function variant(): BelongsTo
    {
        return $this->belongsTo(ProductVariant::class, 'product_variant_id');
    }

    /* ========================= Helpers / Accessors ========================= */

    /**
     * المتاح = الفعلي - المحجوز (لا يسمح بالسالب).
     */
    public function getAvailableAttribute(): int
    {
        return max(0, (int) $this->on_hand - (int) $this->reserved_qty);
    }

    /**
     * هل المخزون منخفض بناءً على min_stock_level المخزَّن في هذا السطر؟
     */
    public function getIsLowAttribute(): bool
    {
        $min = (int) $this->min_stock_level;

        if ($min <= 0) {
            return false;
        }

        return $this->available <= $min;
    }

    /**
     * هل يجب إعادة الطلب بناء على reorder_point؟
     */
    public function getNeedsReorderAttribute(): bool
    {
        $reorderPoint = (int) $this->reorder_point;

        if ($reorderPoint <= 0) {
            return false;
        }

        return $this->available <= $reorderPoint;
    }

    /* ========================= Scopes أساسية ========================= */

    public function scopePositive($q)
    {
        return $q->where('on_hand', '>', 0);
    }

    public function scopeForWarehouse($q, int $warehouseId)
    {
        return $q->where('warehouse_id', $warehouseId);
    }

    public function scopeForVariant($q, int $variantId)
    {
        return $q->where('product_variant_id', $variantId);
    }

    /**
     * أرصدة تخص فرع معيّن (عبر المستودعات المرتبطة بالفرع).
     */
    public function scopeForBranch($q, int $branchId)
    {
        return $q->whereHas('warehouse', function ($w) use ($branchId) {
            $w->whereHas('branches', function ($b) use ($branchId) {
                $b->where('branches.id', $branchId);
            });
        });
    }

    /**
     * ✅ أرصدة تخص الفرع الحالي للمستخدم
     *
     * CRITICAL FIX: يجب أن تبدأ الدالة بـ scope
     */
    public function scopeForCurrentBranch($q)
    {
        $branchId = user_info('branch_id');

        if (!$branchId) {
            return $q; // لو مافي فرع، نرجع كل شي
        }

        return $q->whereHas('warehouse', function ($w) use ($branchId) {
            $w->forBranch($branchId);
        });
    }

    /* ========================= Scopes للتقارير ========================= */

    /**
     * أرصدة لمنتج معيّن (كل نسخه) عبر المستودعات.
     */
    public function scopeForProduct($q, int $productId)
    {
        return $q->whereHas('variant', function ($qq) use ($productId) {
            $qq->where('product_id', $productId);
        });
    }

    /**
     * أرصدة تم تحريكها خلال فترة معيّنة (بناءً على last_movement_at).
     */
    public function scopeMovedBetween($q, $from, $to)
    {
        return $q->whereBetween('last_movement_at', [$from, $to]);
    }

    /**
     * أرصدة ذات مخزون متاح أقل أو يساوي حد معيّن.
     * (يستخدم available = on_hand - reserved_qty)
     */
    public function scopeLowAvailable($q, int $threshold)
    {
        return $q->whereRaw('(on_hand - reserved_qty) <= ?', [$threshold]);
    }

    /**
     * أرصدة أقل من أو عند الحد الأدنى المعرّف في السطر نفسه.
     */
    public function scopeBelowMinLevel($q)
    {
        return $q
            ->where('min_stock_level', '>', 0)
            ->whereRaw('(on_hand - reserved_qty) <= min_stock_level');
    }

    /**
     * أرصدة تحتاج إعادة طلب (available <= reorder_point).
     */
    public function scopeNeedsReorder($q)
    {
        return $q
            ->where('reorder_point', '>', 0)
            ->whereRaw('(on_hand - reserved_qty) <= reorder_point');
    }
}
