<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * شركة الشحن
 */
class ShippingCompany extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'code',
        'phone',
        'email',
        'contact_person',
        'address',
        'notes',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'is_active' => 'boolean',
        ];
    }

    /* ================== العلاقات ================== */

    public function rates(): HasMany
    {
        return $this->hasMany(ShippingCompanyRate::class);
    }

    public function shipments(): HasMany
    {
        return $this->hasMany(Shipment::class);
    }

    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    /* ================== Scopes ================== */

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /* ================== Methods ================== */

    /**
     * الحصول على سعر الشحن لمدينة معينة
     */
    public function getRateForCity(int $cityId): ?ShippingCompanyRate
    {
        return $this->rates()
            ->where('city_id', $cityId)
            ->where('is_active', true)
            ->first();
    }

    /**
     * هل الشحن متاح لهذه المدينة؟
     */
    public function servesCity(int $cityId): bool
    {
        return $this->rates()
            ->where('city_id', $cityId)
            ->where('is_active', true)
            ->exists();
    }

    /**
     * إحصائيات الشحنات
     */
    public function getDeliveredShipmentsCountAttribute(): int
    {
        return $this->shipments()->where('status', 'delivered')->count();
    }

    public function getPendingShipmentsCountAttribute(): int
    {
        return $this->shipments()->whereNotIn('status', ['delivered', 'returned', 'failed'])->count();
    }

    /**
     * إجمالي مبالغ COD المحصلة
     */
    public function getTotalCodCollectedAttribute(): int
    {
        return $this->shipments()->where('cod_collected', true)->sum('cod_amount');
    }

    /**
     * إجمالي مبالغ COD غير المحصلة
     */
    public function getTotalCodPendingAttribute(): int
    {
        return $this->shipments()
            ->where('cod_collected', false)
            ->where('status', 'delivered')
            ->sum('cod_amount');
    }
}
