<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * عنوان الشحن
 */
class ShippingAddress extends Model
{
    use HasFactory;

    protected $fillable = [
        'client_id',
        'city_id',
        'name',
        'phone',
        'address',
        'is_default',
    ];

    protected function casts(): array
    {
        return [
            'is_default' => 'boolean',
        ];
    }

    /* ================== Boot ================== */

    protected static function booted(): void
    {
        // ضمان وجود عنوان افتراضي واحد فقط لكل عميل
        static::saving(function (self $address) {
            if ($address->is_default) {
                static::where('client_id', $address->client_id)
                    ->where('id', '!=', $address->id ?? 0)
                    ->update(['is_default' => false]);
            }
        });

        // إذا تم حذف العنوان الافتراضي، جعل أول عنوان آخر افتراضي
        static::deleted(function (self $address) {
            if ($address->is_default) {
                $firstAddress = static::where('client_id', $address->client_id)->first();
                if ($firstAddress) {
                    $firstAddress->update(['is_default' => true]);
                }
            }
        });
    }

    /* ================== العلاقات ================== */

    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    public function city(): BelongsTo
    {
        return $this->belongsTo(City::class);
    }

    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    /* ================== Scopes ================== */

    public function scopeDefault($query)
    {
        return $query->where('is_default', true);
    }

    public function scopeForClient($query, int $clientId)
    {
        return $query->where('client_id', $clientId);
    }

    /* ================== Accessors ================== */

    /**
     * العنوان الكامل
     */
    public function getFullAddressAttribute(): string
    {
        $parts = array_filter([
            $this->address,
            $this->city?->name,
        ]);

        return implode(', ', $parts);
    }

    /**
     * تنسيق العنوان للعرض
     */
    public function getFormattedAttribute(): string
    {
        return "{$this->name}\n{$this->phone}\n{$this->full_address}";
    }

    /* ================== Methods ================== */

    /**
     * تعيين كعنوان افتراضي
     */
    public function setAsDefault(): void
    {
        $this->update(['is_default' => true]);
    }
}
