<?php

namespace App\Models;

use App\Enums\ShipmentStatus;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * الشحنة
 */
class Shipment extends Model
{
    protected $fillable = [
        'order_id',
        'shipping_company_id',
        'tracking_number',
        'waybill_number',
        'status',
        'shipped_at',
        'delivered_at',
        'returned_at',
        'delivery_attempts',
        'cod_amount',
        'cod_collected',
        'cod_collected_at',
        'notes',
    ];

    protected function casts(): array
    {
        return [
            'status' => ShipmentStatus::class,
            'shipped_at' => 'datetime',
            'delivered_at' => 'datetime',
            'returned_at' => 'datetime',
            'cod_collected_at' => 'datetime',
            'delivery_attempts' => 'integer',
            'cod_amount' => 'integer',
            'cod_collected' => 'boolean',
        ];
    }

    /* ================== Boot ================== */

    protected static function booted(): void
    {
        static::created(function (self $shipment) {
            // تسجيل الحالة الأولى
            $shipment->statusHistory()->create([
                'status' => $shipment->status->value,
                'changed_at' => now(),
            ]);
        });

        static::updating(function (self $shipment) {
            // تسجيل تغيير الحالة
            if ($shipment->isDirty('status')) {
                $shipment->statusHistory()->create([
                    'status' => $shipment->status->value,
                    'changed_at' => now(),
                ]);

                // تحديث التواريخ حسب الحالة
                if ($shipment->status === ShipmentStatus::Picked && ! $shipment->shipped_at) {
                    $shipment->shipped_at = now();
                }

                if ($shipment->status === ShipmentStatus::Delivered && ! $shipment->delivered_at) {
                    $shipment->delivered_at = now();
                }

                if ($shipment->status === ShipmentStatus::Returned && ! $shipment->returned_at) {
                    $shipment->returned_at = now();
                }
            }
        });
    }

    /* ================== العلاقات ================== */

    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    public function shippingCompany(): BelongsTo
    {
        return $this->belongsTo(ShippingCompany::class);
    }

    public function statusHistory(): HasMany
    {
        return $this->hasMany(ShipmentStatusHistory::class);
    }

    /* ================== Scopes ================== */

    public function scopePending($query)
    {
        return $query->whereIn('status', [
            ShipmentStatus::Preparing,
            ShipmentStatus::Picked,
            ShipmentStatus::InTransit,
            ShipmentStatus::OutForDelivery,
        ]);
    }

    public function scopeDelivered($query)
    {
        return $query->where('status', ShipmentStatus::Delivered);
    }

    public function scopeReturned($query)
    {
        return $query->where('status', ShipmentStatus::Returned);
    }

    public function scopeFailed($query)
    {
        return $query->where('status', ShipmentStatus::Failed);
    }

    public function scopeCodPending($query)
    {
        return $query->where('cod_amount', '>', 0)
            ->where('cod_collected', false)
            ->where('status', ShipmentStatus::Delivered);
    }

    /* ================== Accessors ================== */

    /**
     * هل الشحنة نشطة؟
     */
    public function getIsActiveAttribute(): bool
    {
        return $this->status->isActive();
    }

    /**
     * هل الشحنة في حالة نهائية؟
     */
    public function getIsFinalAttribute(): bool
    {
        return $this->status->isFinal();
    }

    /**
     * هل يوجد مبلغ COD معلق؟
     */
    public function getHasPendingCodAttribute(): bool
    {
        return $this->cod_amount > 0
            && ! $this->cod_collected
            && $this->status === ShipmentStatus::Delivered;
    }

    /* ================== Methods ================== */

    /**
     * تحديث الحالة
     */
    public function updateStatus(ShipmentStatus $status, ?string $location = null, ?string $notes = null): void
    {
        $this->update(['status' => $status]);

        // تحديث آخر سجل بالموقع والملاحظات
        $lastHistory = $this->statusHistory()->latest()->first();
        if ($lastHistory) {
            $lastHistory->update([
                'location' => $location,
                'notes' => $notes,
            ]);
        }
    }

    /**
     * تسجيل محاولة توصيل فاشلة
     */
    public function recordFailedAttempt(?string $notes = null): void
    {
        $this->increment('delivery_attempts');

        $this->statusHistory()->create([
            'status' => 'failed_attempt',
            'notes' => $notes,
            'changed_at' => now(),
        ]);
    }

    /**
     * تسجيل تحصيل COD
     */
    public function markCodCollected(): void
    {
        $this->update([
            'cod_collected' => true,
            'cod_collected_at' => now(),
        ]);
    }
}
