<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\DB;

class ReceiptCounter extends Model
{
    use HasFactory;

    protected $fillable = [
        'year',
        'branch_id',
        'current_sequence',
    ];

    protected $casts = [
        'year' => 'integer',
        'current_sequence' => 'integer',
    ];

    /* ==================== العلاقات ==================== */

    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    /* ==================== Scopes ==================== */

    public function scopeForBranch($query, int $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    public function scopeForYear($query, int $year)
    {
        return $query->where('year', $year);
    }

    /* ==================== Helpers ==================== */

    /**
     * توليد رقم إيصال جديد
     *
     * مثال: REC-2025-001, REC-2025-002
     */
    public static function generateReceiptNumber(int $branchId): string
    {
        $year = now('Asia/Baghdad')->year;

        // استخدام معاملة مع قفل للصف لضمان التزامن
        return DB::transaction(function () use ($branchId, $year) {
            $counter = static::forBranch($branchId)
                ->forYear($year)
                ->lockForUpdate()
                ->first();

            if (!$counter) {
                $counter = static::create([
                    'branch_id' => $branchId,
                    'year' => $year,
                    'current_sequence' => 0,
                ]);
            }

            $counter->increment('current_sequence');
            $sequence = $counter->current_sequence;

            return sprintf('REC-%d-%03d', $year, $sequence);
        });
    }
}
