<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\{
    HasMany,
    BelongsTo,
    MorphMany
};

/**
 * PurchaseReturn - مرتجع مشتريات
 *
 * يمكن ربطه بقيود المورد (SupplierLedger)
 * وأيضاً بحركات مخزون (stockMovements) في حال كان المرتجع يؤثر على المخزون.
 */
class PurchaseReturn extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'return_number',
        'purchase_invoice_id',
        'supplier_id',
        'warehouse_id',
        'user_id',
        'return_date',
        'total',
        'status',
        'refund_method',
        'notes',
    ];

    protected $casts = [
        'return_date' => 'date',
        'total'       => 'integer', // المبلغ كعدد صحيح
    ];

    /** ═══════════════════════════════════════════════════════
     *  العلاقات
     * ═══════════════════════════════════════════════════════ */

    public function invoice(): BelongsTo
    {
        return $this->belongsTo(PurchaseInvoice::class, 'purchase_invoice_id');
    }

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(PurchaseReturnItem::class);
    }

    public function ledgerEntries(): MorphMany
    {
        return $this->morphMany(SupplierLedger::class, 'ledgerable');
    }

    /**
     * حركات المخزون المرتبطة بهذا المرتجع كمصدر (PurchaseReturn).
     * مثلاً عند إرجاع بضاعة للمورد (OUT أو IN حسب السياسة).
     */
    public function stockMovements(): MorphMany
    {
        return $this->morphMany(StockMovement::class, 'source');
    }

    /** ═══════════════════════════════════════════════════════
     *  السكوبات
     * ═══════════════════════════════════════════════════════ */

    public function scopeForSupplier($query, int $supplierId)
    {
        return $query->where('supplier_id', $supplierId);
    }

    public function scopeForInvoice($query, int $invoiceId)
    {
        return $query->where('purchase_invoice_id', $invoiceId);
    }

    public function scopeByStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeBetweenDates($query, $from, $to)
    {
        return $query->whereBetween('return_date', [$from, $to]);
    }

    /** ═══════════════════════════════════════════════════════
     *  الميثودات المساعدة
     * ═══════════════════════════════════════════════════════ */

    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    public function isPending(): bool
    {
        return $this->status === 'pending';
    }
}
