<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class PurchaseInvoiceItem extends Model
{
    protected $fillable = [
        'purchase_invoice_id',
        'product_variant_id',
        'quantity_ordered',
        'quantity_received',
        'unit_cost',
        'line_total',
        'receive_status',
        'expiry_date',
        'batch_number',
        'notes',
    ];

    protected $casts = [
        'quantity_ordered' => 'integer',
        'quantity_received' => 'integer',
        'unit_cost' => 'integer',
        'line_total' => 'integer',
        'expiry_date' => 'date',
    ];

    /** ═══════════════════════════════════════════════════════
     *  العلاقات
     * ═══════════════════════════════════════════════════════ */
    public function invoice(): BelongsTo
    {
        return $this->belongsTo(PurchaseInvoice::class, 'purchase_invoice_id');
    }

    public function variant(): BelongsTo
    {
        return $this->belongsTo(ProductVariant::class, 'product_variant_id');
    }

    public function returnItems(): HasMany
    {
        return $this->hasMany(PurchaseReturnItem::class);
    }

    public function variantRows()
    {
        return $this->hasMany(PurchaseInvoiceItemVariant::class, 'purchase_invoice_item_id');
    }

    /** ═══════════════════════════════════════════════════════
     *  السكوبات
     * ═══════════════════════════════════════════════════════ */
    public function scopePending($query)
    {
        return $query->where('receive_status', 'pending');
    }

    public function scopePartiallyReceived($query)
    {
        return $query->where('receive_status', 'partial');
    }

    public function scopeFullyReceived($query)
    {
        return $query->where('receive_status', 'received');
    }

    public function scopeWithBatch($query, string $batchNumber)
    {
        return $query->where('batch_number', $batchNumber);
    }

    /** ═══════════════════════════════════════════════════════
     *  الميثودات المساعدة
     * ═══════════════════════════════════════════════════════ */

    // الكمية المتبقية للاستلام
    public function getRemainingQuantityAttribute(): int
    {
        return max(0, $this->quantity_ordered - $this->quantity_received);
    }

    // نسبة الاستلام
    public function getReceivePercentageAttribute(): float
    {
        if ($this->quantity_ordered == 0) {
            return 0;
        }

        return round(($this->quantity_received / $this->quantity_ordered) * 100, 2);
    }

    // هل تم استلام الكمية بالكامل؟
    public function isFullyReceived(): bool
    {
        return $this->receive_status === 'received';
    }

    // إجمالي الكمية المرتجعة
    public function getTotalReturnedQuantityAttribute(): int
    {
        return (int) $this->returnItems()->sum('quantity_returned');
    }
}
