<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\{
    BelongsTo,
    HasMany,
    MorphMany
};

/**
 * ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
 * 1) PurchaseInvoice - فاتورة المشتريات
 * ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
 *
 * تمثل فاتورة شراء من مورد معيّن، ويمكن ربطها بحركات مخزون
 * عبر العلاقة البوليمورفية stockMovements (source_type / source_id).
 */
class PurchaseInvoice extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'invoice_number',
        'supplier_id',
        'branch_id',
        'warehouse_id',
        'user_id',
        'invoice_date',
        'due_date',
        'subtotal',
        'discount',
        'freight',
        'total',
        'paid',
        'due',
        'status',
        'payment_status',
        'notes',
    ];

    protected $casts = [
        'invoice_date'   => 'date',
        'due_date'       => 'date',

        // مبالغ الفاتورة كأعداد صحيحة (أصغر وحدة عملة)
        'subtotal'       => 'integer',
        'discount'       => 'integer',
        'freight'        => 'integer',
        'total'          => 'integer',
        'paid'           => 'integer',
        'due'            => 'integer',
    ];

    /** ═══════════════════════════════════════════════════════
     *  العلاقات (Relations)
     * ═══════════════════════════════════════════════════════ */

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(PurchaseInvoiceItem::class);
    }

    public function returns(): HasMany
    {
        return $this->hasMany(PurchaseReturn::class);
    }

    public function ledgerEntries(): MorphMany
    {
        return $this->morphMany(SupplierLedger::class, 'ledgerable');
    }

    public function paymentAllocations(): HasMany
    {
        return $this->hasMany(PaymentAllocation::class);
    }

    /**
     * حركات المخزون المرتبطة بهذه الفاتورة كمصدر (PurchaseInvoice).
     * تعتمد على حقول source_type / source_id في جدول stock_movements.
     */
    public function stockMovements(): MorphMany
    {
        return $this->morphMany(StockMovement::class, 'source');
    }

    /** ═══════════════════════════════════════════════════════
     *  السكوبات (Scopes)
     * ═══════════════════════════════════════════════════════ */

    public function scopeForSupplier($query, int $supplierId)
    {
        return $query->where('supplier_id', $supplierId);
    }

    public function scopeForBranch($query, int $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    public function scopeForWarehouse($query, int $warehouseId)
    {
        return $query->where('warehouse_id', $warehouseId);
    }

    public function scopeByStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    public function scopeByPaymentStatus($query, string $paymentStatus)
    {
        return $query->where('payment_status', $paymentStatus);
    }

    public function scopeUnpaid($query)
    {
        return $query->whereIn('payment_status', ['unpaid', 'partial']);
    }

    public function scopeOverdue($query)
    {
        return $query
            ->where('due_date', '<', now())
            ->whereIn('payment_status', ['unpaid', 'partial']);
    }

    public function scopeBetweenDates($query, $from, $to)
    {
        return $query->whereBetween('invoice_date', [$from, $to]);
    }

    /** ═══════════════════════════════════════════════════════
     *  الأكسسوارات والميثودات المساعدة
     * ═══════════════════════════════════════════════════════ */

    // هل الفاتورة مدفوعة بالكامل؟
    public function isFullyPaid(): bool
    {
        return $this->payment_status === 'paid';
    }

    // هل الفاتورة متأخرة؟
    public function isOverdue(): bool
    {
        return $this->due_date
            && $this->due_date->isPast()
            && in_array($this->payment_status, ['unpaid', 'partial']);
    }

    // هل الفاتورة مستلمة بالكامل (من ناحية المخزون)؟
    public function isFullyReceived(): bool
    {
        return $this->status === 'received';
    }

    // نسبة الاستلام
    public function getReceivePercentageAttribute(): float
    {
        $totalOrdered = $this->items()->sum('quantity_ordered');
        if ($totalOrdered == 0) {
            return 0;
        }

        $totalReceived = $this->items()->sum('quantity_received');

        return round(($totalReceived / $totalOrdered) * 100, 2);
    }

    // نسبة الدفع
    public function getPaymentPercentageAttribute(): float
    {
        if ($this->total == 0) {
            return 0;
        }

        return round(($this->paid / $this->total) * 100, 2);
    }
}
