<?php

namespace App\Models;

use App\Enums\PromotionAppliesTo;
use App\Enums\PromotionType;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * العروض والخصومات
 */
class Promotion extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'type',
        'discount_value',
        'buy_quantity',
        'get_quantity',
        'min_purchase_amount',
        'max_discount_amount',
        'applies_to',
        'applicable_ids',
        'priority',
        'is_stackable',
        'usage_limit',
        'usage_count',
        'starts_at',
        'ends_at',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'type' => PromotionType::class,
            'applies_to' => PromotionAppliesTo::class,
            'discount_value' => 'decimal:2',
            'buy_quantity' => 'integer',
            'get_quantity' => 'integer',
            'min_purchase_amount' => 'integer',
            'max_discount_amount' => 'integer',
            'applicable_ids' => 'array',
            'priority' => 'integer',
            'is_stackable' => 'boolean',
            'usage_limit' => 'integer',
            'usage_count' => 'integer',
            'starts_at' => 'datetime',
            'ends_at' => 'datetime',
            'is_active' => 'boolean',
        ];
    }

    /* ================== العلاقات ================== */

    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    public function orderItems(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    /* ================== Scopes ================== */

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeValid($query)
    {
        $now = now();

        return $query->where('is_active', true)
            ->where(function ($q) use ($now) {
                $q->whereNull('starts_at')
                    ->orWhere('starts_at', '<=', $now);
            })
            ->where(function ($q) use ($now) {
                $q->whereNull('ends_at')
                    ->orWhere('ends_at', '>=', $now);
            })
            ->where(function ($q) {
                $q->whereNull('usage_limit')
                    ->orWhereColumn('usage_count', '<', 'usage_limit');
            });
    }

    public function scopeByPriority($query)
    {
        return $query->orderBy('priority', 'desc');
    }

    /* ================== Accessors ================== */

    /**
     * هل العرض ساري المفعول؟
     */
    public function getIsValidAttribute(): bool
    {
        if (! $this->is_active) {
            return false;
        }

        $now = now();

        if ($this->starts_at && $this->starts_at->isFuture()) {
            return false;
        }

        if ($this->ends_at && $this->ends_at->isPast()) {
            return false;
        }

        if ($this->usage_limit && $this->usage_count >= $this->usage_limit) {
            return false;
        }

        return true;
    }

    /**
     * الاستخدامات المتبقية
     */
    public function getRemainingUsageAttribute(): ?int
    {
        if ($this->usage_limit === null) {
            return null;
        }

        return max(0, $this->usage_limit - $this->usage_count);
    }

    /* ================== Methods ================== */

    /**
     * هل ينطبق العرض على منتج معين؟
     */
    public function appliesTo(Product $product): bool
    {
        return match ($this->applies_to) {
            PromotionAppliesTo::All => true,
            PromotionAppliesTo::Product => in_array($product->id, $this->applicable_ids ?? []),
            PromotionAppliesTo::Category => in_array($product->category_id, $this->applicable_ids ?? []),
            PromotionAppliesTo::Brand => in_array($product->brand_id, $this->applicable_ids ?? []),
            default => false,
        };
    }

    /**
     * حساب مبلغ الخصم
     */
    public function calculateDiscount(int $amount): int
    {
        if (! $this->is_valid) {
            return 0;
        }

        // التحقق من الحد الأدنى للشراء
        if ($this->min_purchase_amount && $amount < $this->min_purchase_amount) {
            return 0;
        }

        $discount = match ($this->type) {
            PromotionType::Percentage => (int) round($amount * ($this->discount_value / 100)),
            PromotionType::Fixed => (int) $this->discount_value,
            PromotionType::FreeShipping => 0, // يُعالج منفصلاً
            PromotionType::MinPurchaseDiscount => (int) $this->discount_value,
            default => 0,
        };

        // تطبيق الحد الأقصى للخصم
        if ($this->max_discount_amount && $discount > $this->max_discount_amount) {
            $discount = $this->max_discount_amount;
        }

        return $discount;
    }

    /**
     * زيادة عداد الاستخدام
     */
    public function incrementUsage(): void
    {
        $this->increment('usage_count');
    }
}
