<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\{
    BelongsTo,
    BelongsToMany,
    HasMany
};
use Illuminate\Database\Eloquent\Casts\Attribute as CastAttribute;

/**
 * نسخة المنتج (Variant)
 * - تحمل كلفة أساسية (cost)
 * - ترتبط بمخزون عبر StockBalance لكل مستودع
 * - تدعم خصائص (حجم، لون، ...إلخ) عبر AttributeValue
 */
class ProductVariant extends Model
{
    use SoftDeletes;

    protected $table = 'product_variants';

    protected $appends = [
        'variant_label',
        'total_on_hand',
        'total_reserved',
        'total_available',
        'full_name',
    ];

    // تحميل الخصائص دائماً مع المتغير لتفادي N+1
    protected $with = ['attributeValues.attribute'];

    protected $fillable = [
        'product_id',
        'sku',
        'cost',             // BIGINT (أصغر وحدة عملة)
        'is_active',
        'is_default',
        'options_hash',     // بصمة تركيبة الخصائص
        'min_stock_level',  // حد أدنى على مستوى المتغير ككل
    ];

    protected $casts = [
        'cost'            => 'integer',
        'is_active'       => 'boolean',
        'is_default'      => 'boolean',
        'min_stock_level' => 'integer',
    ];

    /* ====================== العلاقات ====================== */

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function barcodes(): HasMany
    {
        return $this->hasMany(Barcode::class);
    }

    public function attributeValues(): BelongsToMany
    {
        return $this->belongsToMany(
            AttributeValue::class,
            'variant_attribute_values',
            'variant_id',
            'attribute_value_id'
        )->withPivot('attribute_id')->withTimestamps();
    }

    /** علاقات الجرد */

    public function stockBalances(): HasMany
    {
        return $this->hasMany(StockBalance::class, 'product_variant_id');
    }

    public function stockMovements(): HasMany
    {
        return $this->hasMany(StockMovement::class, 'product_variant_id');
    }

    /** علاقة الأسعار الخاصة بهذه النسخة */

    public function prices(): HasMany
    {
        return $this->hasMany(ProductPrice::class, 'product_variant_id');
    }

    /** باركود رئيسي */

    public function getPrimaryBarcodeAttribute(): ?string
    {
        return $this->barcodes()
            ->where('is_primary', true)
            ->where('is_active', true)
            ->value('code');
    }

    /* ====================== سكوبات ====================== */

    public function scopeActive($q)
    {
        return $q->where('is_active', true);
    }

    /**
     * المتغيرات ذات المخزون المنخفض.
     * نعتمد هنا على:
     *  SUM(on_hand - reserved_qty) <= product_variants.min_stock_level
     */
    public function scopeLowStock($q)
    {
        $sub = StockBalance::query()
            ->selectRaw('SUM(on_hand - reserved_qty)')
            ->whereColumn('product_variant_id', 'product_variants.id');

        return $q
            ->where('min_stock_level', '>', 0)
            ->whereRaw(
                'COALESCE((' . $sub->toSql() . '), 0) <= product_variants.min_stock_level',
                $sub->getBindings()
            );
    }

    /* ====================== Hooks: SKU + options_hash + افتراضية ====================== */

    protected static function booted()
    {
        static::creating(function (self $v) {
            // توليد SKU تلقائي في حال عدم التحديد
            if (empty($v->sku)) {
                $v->sku = self::generateSku((int) $v->product_id);
            }
        });

        static::saving(function (self $v) {
            // حساب بصمة تركيبة الخصائص
            $ids = $v->exists
                ? $v->attributeValues()->pluck('attribute_values.id')->sort()->values()->all()
                : [];

            $v->options_hash = $ids ? hash('sha256', json_encode($ids)) : null;

            // ضمان وجود متغير افتراضي واحد فقط لكل منتج
            if ($v->is_default) {
                static::where('product_id', $v->product_id)
                    ->where('id', '!=', $v->id ?? 0)
                    ->update(['is_default' => false]);
            }
        });

        static::created(function (self $v) {
            // إعادة حساب الـ hash بعد الإنشاء في حال إضافة الخصائص بعد الـ create
            $ids  = $v->attributeValues()->pluck('attribute_values.id')->sort()->values()->all();
            $hash = $ids ? hash('sha256', json_encode($ids)) : null;

            if ($hash !== $v->options_hash) {
                $v->options_hash = $hash;
                $v->saveQuietly();
            }
        });
    }

    private static function generateSku(int $productId): string
    {
        $prefix      = 'PRD';
        $productPart = str_pad((string) $productId, 4, '0', STR_PAD_LEFT);
        $seq         = static::withTrashed()->where('product_id', $productId)->count() + 1;

        do {
            $variantPart = str_pad((string) $seq, 3, '0', STR_PAD_LEFT);
            $sku         = "{$prefix}-{$productPart}-{$variantPart}";
            $seq++;
        } while (static::withTrashed()->where('sku', $sku)->exists());

        return $sku;
    }

    /* ====================== Accessors / Attributes ====================== */

    /**
     * وسم وصفي للمتغير يجمع اسم الخاصية + القيمة.
     */
    protected function variantLabel(): CastAttribute
    {
        return CastAttribute::make(
            get: function () {
                $values = $this->attributeValues
                    ->loadMissing('attribute')
                    ->map(function ($val) {
                        $attr = $val->attribute?->display_name ?? 'Attr';
                        $name = $val->display_value;
                        return "{$attr} - {$name}";
                    })
                    ->filter()
                    ->values()
                    ->all();

                return $values ? implode(' / ', $values) : ($this->sku ?? "Variant #{$this->id}");
            }
        );
    }

    /**
     * الاسم الكامل (اسم المنتج + قيم الخصائص).
     */
    public function getFullNameAttribute(): string
    {
        $baseName   = $this->product?->name ?? '';
        $attributes = $this->attributeValues?->pluck('display_value')->join(' - ');

        if ($attributes) {
            return "{$baseName} ({$attributes})";
        }

        return $baseName ?: 'متغير بدون اسم';
    }

    /** مخزون محسوب من stock_balances */

    public function getTotalOnHandAttribute(): int
    {
        return (int) $this->stockBalances()->sum('on_hand');
    }

    public function getTotalReservedAttribute(): int
    {
        return (int) $this->stockBalances()->sum('reserved_qty');
    }

    public function getTotalAvailableAttribute(): int
    {
        $on  = $this->total_on_hand;
        $res = $this->total_reserved;

        return max(0, (int) ($on - $res));
    }

    /**
     * حساب سعر البيع النهائي حسب نوع العميل:
     * - retail (مفرد)
     * - wholesale (جملة)
     * - agent (وكيل)
     *
     * المنطق:
     * 1) نبحث عن ProductPrice مربوط بهذه النسخة (product_variant_id)
     * 2) لو ماكو، نرجع لسعر عام على مستوى المنتج (product_id فقط)
     * 3) لو ماكو، نستخدم cost من الـ Variant نفسه كـ base
     */
    public function getPriceForType(string $type = 'retail'): int
    {
        // 1) سعر خاص لهذه النسخة
        $priceModel = $this->prices()
            ->active()
            ->first();

        // 2) سعر عام على مستوى المنتج (بدون ربط بمتغير معيّن)
        if (! $priceModel) {
            $priceModel = ProductPrice::query()
                ->where('product_id', $this->product_id)
                ->whereNull('product_variant_id')
                ->active()
                ->first();
        }

        // base = cost (لا نستخدم حقل price افتراضاً)
        $base = (int) ($this->cost ?? 0);

        if (! $priceModel) {
            return $base;
        }

        $fixedField  = "{$type}_price";   // retail_price / wholesale_price / agent_price
        $marginField = "{$type}_margin";  // retail_margin / wholesale_margin / agent_margin

        if ($priceModel->{$fixedField} > 0) {
            return (int) $priceModel->{$fixedField};
        }

        $margin = (float) ($priceModel->{$marginField} ?? 0);

        return (int) round($base * (1 + $margin / 100));
    }
}
