<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ProductPrice extends Model
{
    use HasFactory;

    public const STATUS_ACTIVE = 1;

    public const STATUS_INACTIVE = 0;

    protected $table = 'product_prices';

    protected $fillable = [
        'product_id',
        'product_variant_id',
        'retail_margin',
        'wholesale_margin',
        'agent_margin',
        'retail_price',
        'wholesale_price',
        'agent_price',
        'status',
        'is_on_sale',
        'discount_percentage',
        'sale_price',
        'sale_start_date',
        'sale_end_date',
    ];

    protected $casts = [
        'status' => 'boolean',
        'retail_margin' => 'float',
        'wholesale_margin' => 'float',
        'agent_margin' => 'float',
        'retail_price' => 'integer',
        'wholesale_price' => 'integer',
        'agent_price' => 'integer',
        'is_on_sale' => 'boolean',
        'discount_percentage' => 'integer',
        'sale_price' => 'integer',
        'sale_start_date' => 'datetime',
        'sale_end_date' => 'datetime',
    ];

    // إضافة unit_cost للـ appends إذا أردت أن يظهر تلقائياً في JSON
    // protected $appends = ['unit_cost'];

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function variant(): BelongsTo
    {
        return $this->belongsTo(ProductVariant::class, 'product_variant_id');
    }

    public function scopeActive($query)
    {
        return $query->where('status', self::STATUS_ACTIVE);
    }

    public function isActive(): bool
    {
        return $this->status === self::STATUS_ACTIVE;
    }

    /**
     * التحقق من أن العرض نشط وضمن الفترة الزمنية المحددة
     */
    public function isOnSale(): bool
    {
        if (! $this->is_on_sale) {
            return false;
        }

        $now = now();

        if ($this->sale_start_date && $now->lt($this->sale_start_date)) {
            return false;
        }

        if ($this->sale_end_date && $now->gt($this->sale_end_date)) {
            return false;
        }

        return true;
    }

    /**
     * الحصول على السعر النهائي (مع العرض إن وُجد)
     */
    public function getFinalPrice(string $type = 'retail'): int
    {
        if ($this->isOnSale() && $this->sale_price > 0) {
            return (int) $this->sale_price;
        }

        $priceField = "{$type}_price";

        return (int) ($this->{$priceField} ?? 0);
    }

    /**
     * Scope للحصول على المنتجات التي عليها عروض نشطة
     */
    public function scopeOnSale($query)
    {
        $now = now();

        return $query->where('is_on_sale', true)
            ->where(function ($q) use ($now) {
                $q->whereNull('sale_start_date')
                    ->orWhere('sale_start_date', '<=', $now);
            })
            ->where(function ($q) use ($now) {
                $q->whereNull('sale_end_date')
                    ->orWhere('sale_end_date', '>=', $now);
            });
    }

    /**
     * الحصول على سعر الكلفة من آخر حركة مخزون
     */
    protected function unitCost(): Attribute
    {
        return Attribute::make(
            get: function () {
                if (! $this->product_variant_id) {
                    return null;
                }

                return StockMovement::query()
                    ->where('product_variant_id', $this->product_variant_id)
                    ->orderByDesc('id')
                    ->value('unit_cost');
            }
        );
    }
}
