<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;

class ProductModel extends Model implements HasMedia
{
    use InteractsWithMedia;
    use SoftDeletes;

    public const MEDIA_COLLECTION_PREVIEW = 'model_preview';

    public const MEDIA_COLLECTION_GLB = 'model_glb';

    protected $fillable = [
        'product_id',
        'name',
        'description',
        'glb_file_path',
        'available_options',
        'metadata',
        'is_active',
        'is_featured',
        'order',
        'views_count',
        'designs_count',
    ];

    protected $casts = [
        'available_options' => 'array',
        'metadata' => 'array',
        'is_active' => 'boolean',
        'is_featured' => 'boolean',
        'views_count' => 'integer',
        'designs_count' => 'integer',
    ];

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection(self::MEDIA_COLLECTION_PREVIEW)
            ->useDisk('upload');

        $this->addMediaCollection(self::MEDIA_COLLECTION_GLB)
            ->useDisk('upload')
            ->singleFile();
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function designs(): HasMany
    {
        return $this->hasMany(ProductDesign::class);
    }

    public function publicDesigns(): HasMany
    {
        return $this->designs()->where('is_public', true);
    }

    public function featuredDesigns(): HasMany
    {
        return $this->designs()->where('is_featured', true);
    }

    public function incrementViews(): void
    {
        $this->increment('views_count');
    }

    public function incrementDesigns(): void
    {
        $this->increment('designs_count');
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopePopular($query)
    {
        return $query->orderByDesc('designs_count');
    }

    public function getGlbUrlAttribute(): ?string
    {
        return $this->getFirstMediaUrl(self::MEDIA_COLLECTION_GLB);
    }

    public function getPreviewImageUrlAttribute(): ?string
    {
        return $this->getFirstMediaUrl(self::MEDIA_COLLECTION_PREVIEW);
    }
}
