<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;

class ProductDesign extends Model implements HasMedia
{
    use InteractsWithMedia;
    use SoftDeletes;

    public const MEDIA_COLLECTION_PREVIEW = 'design_preview';

    public const MEDIA_COLLECTION_ASSETS = 'design_assets';

    protected $fillable = [
        'product_model_id',
        'user_id',
        'client_id',
        'order_item_id',
        'name',
        'configuration',
        'preview_image',
        'is_public',
        'is_featured',
        'views_count',
        'likes_count',
        'purchases_count',
    ];

    protected $casts = [
        'configuration' => 'array',
        'is_public' => 'boolean',
        'is_featured' => 'boolean',
        'views_count' => 'integer',
        'likes_count' => 'integer',
        'purchases_count' => 'integer',
    ];

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection(self::MEDIA_COLLECTION_PREVIEW)
            ->useDisk('upload')
            ->singleFile();

        $this->addMediaCollection(self::MEDIA_COLLECTION_ASSETS)
            ->useDisk('upload');
    }

    public function productModel(): BelongsTo
    {
        return $this->belongsTo(ProductModel::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    public function orderItem(): BelongsTo
    {
        return $this->belongsTo(InvoiceItem::class, 'order_item_id');
    }

    public function likedBy(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'product_design_likes')
            ->withPivot('created_at');
    }

    public function isLikedBy(?User $user): bool
    {
        if (! $user) {
            return false;
        }

        return $this->likedBy()->where('user_id', $user->id)->exists();
    }

    public function toggleLike(User $user): void
    {
        if ($this->isLikedBy($user)) {
            $this->likedBy()->detach($user->id);
            $this->decrement('likes_count');
        } else {
            $this->likedBy()->attach($user->id);
            $this->increment('likes_count');
        }
    }

    public function incrementViews(): void
    {
        $this->increment('views_count');
    }

    public function incrementPurchases(): void
    {
        $this->increment('purchases_count');
    }

    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopePopular($query)
    {
        return $query->orderByDesc('purchases_count');
    }

    public function scopeMostLiked($query)
    {
        return $query->orderByDesc('likes_count');
    }

    public function getPreviewImageUrlAttribute(): ?string
    {
        if ($this->preview_image) {
            return asset('storage/'.$this->preview_image);
        }

        return $this->getFirstMediaUrl(self::MEDIA_COLLECTION_PREVIEW);
    }
}
