<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ProductCard extends Model
{
    protected $table = 'product_cards';

    protected $fillable = [
        'branch_id',
        'user_id',
        'product_id',
        'product_variant_id',
        'title',

        // جديد
        'description',
        'notes',

        'price',
        'show_discount',
        'discount_percent',
        'is_active',
    ];

    protected $casts = [
        'branch_id' => 'integer',
        'user_id' => 'integer',
        'product_id' => 'integer',
        'product_variant_id' => 'integer',

        'description' => 'string',
        'notes' => 'string',

        'price' => 'integer',
        'show_discount' => 'boolean',
        'discount_percent' => 'integer',
        'is_active' => 'boolean',
    ];

    /* ---------------- العلاقات ---------------- */

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function variant(): BelongsTo
    {
        return $this->belongsTo(ProductVariant::class, 'product_variant_id');
    }

    /* ---------------- Accessors ---------------- */

    public function getFormattedPriceAttribute(): string
    {
        return number_format((int) ($this->price ?? 0)).' د.ع';
    }

    public function getCleanDescriptionAttribute(): string
    {
        return trim((string) ($this->description ?? ''));
    }

    public function getCleanNotesAttribute(): string
    {
        return trim((string) ($this->notes ?? ''));
    }

    public function getHasDiscountAttribute(): bool
    {
        $percent = (int) ($this->discount_percent ?? 0);

        return (bool) $this->show_discount && $percent > 0;
    }

    /**
     * يستخدم صورة المنتج (وليس للكارت صورة مستقلة).
     */
    public function getImageUrlAttribute(): ?string
    {
        $url = $this->product?->image_url;

        return $this->encodeUrl($url);
    }

    /**
     * يستخدم صورة المنتج المصغّرة إن وجدت، وإلا يرجع الأصلية.
     */
    public function getImageThumbUrlAttribute(): ?string
    {
        $url = $this->product?->image_thumb_url ?? $this->product?->image_url;

        return $this->encodeUrl($url);
    }

    private function encodeUrl(?string $url): ?string
    {
        if (! $url) {
            return null;
        }

        // يعالج المسافات/العربي داخل URL بدون كسر البروتوكول والـ slashes
        return preg_replace_callback(
            '/[^A-Za-z0-9\-._~:\/?#\[\]@!$&\'()*+,;=%]/u',
            fn ($m) => rawurlencode($m[0]),
            $url
        );
    }
}
