<?php

namespace App\Models;

use App\Enums\ProductUnit;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;

class Product extends Model implements HasMedia
{
    use HasSlug;
    use InteractsWithMedia;
    use SoftDeletes;
    /* ====================== Media Collections ====================== */

    public const MEDIA_COLLECTION_THUMBNAIL = 'product_thumbnail';

    public const MEDIA_COLLECTION_GALLERY = 'product_gallery';

    protected $fillable = [
        'name',
        'description',
        'category_id',
        'brand_id',
        'unit',
        'is_variable',
        'is_active',
    ];

    protected $casts = [
        'unit' => ProductUnit::class,
        'is_variable' => 'boolean',
        'is_active' => 'boolean',
    ];

    protected $appends = [
        'thumbnail_url',
        'thumbnail_thumb_url',
        'gallery_urls',
        'gallery_thumb_urls',
    ];

    /* ====================== Boot ====================== */

    protected static function booted(): void
    {
        static::deleting(function (Product $product) {
            $variants = $product->variants()->withTrashed()->get();

            foreach ($variants as $variant) {
                $usesSoftDeletes = in_array(SoftDeletes::class, class_uses_recursive($variant));

                if ($product->isForceDeleting()) {
                    $usesSoftDeletes ? $variant->forceDelete() : $variant->delete();
                } else {
                    $usesSoftDeletes
                        ? $variant->delete()
                        : $variant->update(['is_active' => false]);
                }
            }
        });

        static::restoring(function (Product $product) {
            $variants = $product->variants()->withTrashed()->get();

            foreach ($variants as $variant) {
                $usesSoftDeletes = in_array(SoftDeletes::class, class_uses_recursive($variant));

                if ($usesSoftDeletes && method_exists($variant, 'restore')) {
                    $variant->restore();
                } else {
                    $variant->updateQuietly(['is_active' => true]);
                }
            }
        });
    }

    /* ====================== Scopes ====================== */

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /* ====================== Slug ====================== */
    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('name')
            ->saveSlugsTo('slug');
    }

    /* ====================== Media Library ====================== */

    public function registerMediaCollections(): void
    {
        // Thumbnail (صورة واحدة)
        $this->addMediaCollection(self::MEDIA_COLLECTION_THUMBNAIL)
            ->useDisk('upload')
            ->singleFile();

        // Gallery (عدة صور)
        $this->addMediaCollection(self::MEDIA_COLLECTION_GALLERY)
            ->useDisk('upload');
    }

    public function registerMediaConversions(?Media $media = null): void
    {
        $this->addMediaConversion('thumb')
            ->width(600)
            ->height(600)
            ->sharpen(10)
            ->nonQueued();
    }

    private function encodeUrl(?string $url): ?string
    {
        if (! $url) {
            return null;
        }

        return preg_replace_callback(
            '/[^A-Za-z0-9\-._~:\/?#\[\]@!$&\'()*+,;=%]/u',
            fn ($m) => rawurlencode($m[0]),
            $url
        );
    }

    /* ====================== Accessors ====================== */

    public function getThumbnailUrlAttribute(): ?string
    {
        $url = $this->getFirstMediaUrl(self::MEDIA_COLLECTION_THUMBNAIL);

        return $url !== '' ? $this->encodeUrl($url) : null;
    }

    public function getThumbnailThumbUrlAttribute(): ?string
    {
        $url = $this->getFirstMediaUrl(self::MEDIA_COLLECTION_THUMBNAIL, 'thumb');

        return $url !== '' ? $this->encodeUrl($url) : $this->thumbnail_url;
    }

    public function getGalleryUrlsAttribute(): array
    {
        return $this->getMedia(self::MEDIA_COLLECTION_GALLERY)
            ->map(fn (Media $m) => $this->encodeUrl($m->getUrl()))
            ->values()
            ->all();
    }

    public function getGalleryThumbUrlsAttribute(): array
    {
        return $this->getMedia(self::MEDIA_COLLECTION_GALLERY)
            ->map(fn (Media $m) => $this->encodeUrl($m->getUrl('thumb')))
            ->values()
            ->all();
    }

    /* ====================== Relations ====================== */

    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    public function variants(): HasMany
    {
        return $this->hasMany(ProductVariant::class);
    }

    public function defaultVariant(): HasOne
    {
        return $this->hasOne(ProductVariant::class)->where('is_default', true);
    }

    /**
     * المتغيرات للمنتج المتغير (بدون النسخة الأساسية)
     */
    public function variableVariants(): HasMany
    {
        return $this->hasMany(ProductVariant::class)->where('is_default', false);
    }

    /**
     * جلب النسخة الأساسية للمنتج البسيط
     */
    public function getBaseVariant(): ?ProductVariant
    {
        return $this->variants()->where('is_default', true)->first()
            ?? $this->variants()->first();
    }

    public function prices(): HasMany
    {
        return $this->hasMany(ProductPrice::class);
    }

    public function models(): HasMany
    {
        return $this->hasMany(ProductModel::class);
    }

    public function activeModels(): HasMany
    {
        return $this->models()->where('is_active', true);
    }

    /* ====================== Stock Helpers ====================== */

    public function getTotalStockAttribute(): int
    {
        if (! class_exists(\App\Models\StockBalance::class)) {
            return 0;
        }

        $variantIds = $this->variants()->pluck('id');

        return $variantIds->isEmpty()
            ? 0
            : (int) \App\Models\StockBalance::whereIn('product_variant_id', $variantIds)->sum('on_hand');
    }

    public function isInStock(): bool
    {
        return $this->total_stock > 0;
    }
}
