<?php

namespace App\Models;

use App\Models\Concerns\HasBranch;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class PayrollRun extends Model
{
    use HasFactory, HasBranch, SoftDeletes;

    protected $fillable = [
        'branch_id',
        'user_id',          // 👈 المستخدم الذي قام بالصرف
        'year',
        'month',
        'is_locked',
        'locked_at',
    ];

    protected $casts = [
        'branch_id' => 'integer',
        'user_id'   => 'integer',
        'year'      => 'integer',
        'month'     => 'integer',
        'is_locked' => 'boolean',
        'locked_at' => 'datetime',
    ];

    /* =========================
     * العلاقات
     * ========================= */

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function payrolls(): HasMany
    {
        return $this->hasMany(Payroll::class);
    }

    /* =========================
     * Helpers
     * ========================= */

    public function getLabelAttribute(): string
    {
        return sprintf('%04d / %02d', $this->year, $this->month);
    }

    public function lock(): void
    {
        $this->update([
            'is_locked' => true,
            'locked_at' => now(),
        ]);
    }

    public function unlock(): void
    {
        $this->update([
            'is_locked' => false,
            'locked_at' => null,
        ]);
    }
}
