<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class OvertimeRecord extends Model
{
    protected $fillable = [
        'branch_id',
        'employee_id',
        'attendance_id',
        'work_schedule_id',
        'date',
        'overtime_minutes',
        'overtime_amount',
        'is_approved',
        'is_paid',
        'approved_at',
        'approved_by',
        'paid_at',
        'notes',
    ];

    protected $casts = [
        'date' => 'date',
        'overtime_minutes' => 'integer',
        'overtime_amount' => 'integer',
        'is_approved' => 'boolean',
        'is_paid' => 'boolean',
        'approved_at' => 'datetime',
        'paid_at' => 'datetime',
    ];

    // ==================== العلاقات ====================

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    public function attendance(): BelongsTo
    {
        return $this->belongsTo(Attendance::class);
    }

    public function workSchedule(): BelongsTo
    {
        return $this->belongsTo(WorkSchedule::class);
    }

    public function approvedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    // ⚠️ تم إزالة علاقة payrollItems لأن الجدول غير موجود بعد
    // يمكنك إضافتها لاحقاً عند إنشاء نظام الرواتب

    // ==================== Accessors ====================

    /**
     * ساعات الوقت الإضافي
     */
    public function getOvertimeHoursAttribute(): float
    {
        return round($this->overtime_minutes / 60, 2);
    }

    /**
     * هل قيد الانتظار؟
     */
    public function getIsPendingAttribute(): bool
    {
        return !$this->is_approved && !$this->is_paid;
    }

    /**
     * حالة السجل
     */
    public function getStatusAttribute(): string
    {
        if ($this->is_paid) {
            return 'paid';
        }

        if ($this->is_approved) {
            return 'approved';
        }

        return 'pending';
    }

    /**
     * نص الحالة بالعربي
     */
    public function getStatusLabelAttribute(): string
    {
        return match ($this->status) {
            'paid' => 'مدفوع',
            'approved' => 'معتمد',
            'pending' => 'قيد الانتظار',
            default => 'غير معروف',
        };
    }

    /**
     * لون الحالة
     */
    public function getStatusColorAttribute(): string
    {
        return match ($this->status) {
            'paid' => 'success',
            'approved' => 'info',
            'pending' => 'warning',
            default => 'gray',
        };
    }

    /**
     * أيقونة الحالة
     */
    public function getStatusIconAttribute(): string
    {
        return match ($this->status) {
            'paid' => 'heroicon-o-check-circle',
            'approved' => 'heroicon-o-clock',
            'pending' => 'heroicon-o-exclamation-triangle',
            default => 'heroicon-o-question-mark-circle',
        };
    }

    /**
     * المبلغ منسق
     */
    public function getFormattedAmountAttribute(): string
    {
        return number_format($this->overtime_amount) . ' د.ع';
    }

    /**
     * الساعات منسقة
     */
    public function getFormattedHoursAttribute(): string
    {
        return $this->overtime_hours . ' ساعة';
    }

    /**
     * التاريخ منسق
     */
    public function getFormattedDateAttribute(): string
    {
        return $this->date->format('Y-m-d');
    }

    // ==================== دوال العمليات ====================

    /**
     * الموافقة على الوقت الإضافي
     */
    public function approve(?int $approvedBy = null): void
    {
        $this->update([
            'is_approved' => true,
            'approved_at' => now(),
            'approved_by' => $approvedBy ?? user_info('id'),
        ]);

        // تحديث سجل الحضور
        $this->attendance?->update([
            'overtime_approved' => true,
        ]);
    }

    /**
     * إلغاء الموافقة
     */
    public function unapprove(): void
    {
        $this->update([
            'is_approved' => false,
            'approved_at' => null,
            'approved_by' => null,
        ]);

        // تحديث سجل الحضور
        $this->attendance?->update([
            'overtime_approved' => false,
        ]);
    }

    /**
     * تأشير كمدفوع
     */
    public function markAsPaid(): void
    {
        $this->update([
            'is_paid' => true,
            'paid_at' => now(),
        ]);
    }

    /**
     * إلغاء الدفع
     */
    public function markAsUnpaid(): void
    {
        $this->update([
            'is_paid' => false,
            'paid_at' => null,
        ]);
    }

    /**
     * الموافقة والدفع مباشرة
     */
    public function approveAndPay(?int $approvedBy = null): void
    {
        $this->update([
            'is_approved' => true,
            'approved_at' => now(),
            'approved_by' => $approvedBy ?? user_info('id'),
            'is_paid' => true,
            'paid_at' => now(),
        ]);

        $this->attendance?->update([
            'overtime_approved' => true,
        ]);
    }

    // ==================== Scopes ====================

    /**
     * لموظف معين
     */
    public function scopeForEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    /**
     * لفرع معين
     */
    public function scopeForBranch($query, $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    /**
     * لتاريخ معين
     */
    public function scopeForDate($query, $date)
    {
        return $query->whereDate('date', $date);
    }

    /**
     * لشهر معين
     */
    public function scopeForMonth($query, int $year, int $month)
    {
        return $query->whereYear('date', $year)
            ->whereMonth('date', $month);
    }

    /**
     * بين تاريخين
     */
    public function scopeBetweenDates($query, $startDate, $endDate)
    {
        return $query->whereBetween('date', [$startDate, $endDate]);
    }

    /**
     * قيد الانتظار
     */
    public function scopePending($query)
    {
        return $query->where('is_approved', false)
            ->where('is_paid', false);
    }

    /**
     * معتمد
     */
    public function scopeApproved($query)
    {
        return $query->where('is_approved', true);
    }

    /**
     * غير معتمد
     */
    public function scopeNotApproved($query)
    {
        return $query->where('is_approved', false);
    }

    /**
     * مدفوع
     */
    public function scopePaid($query)
    {
        return $query->where('is_paid', true);
    }

    /**
     * غير مدفوع
     */
    public function scopeUnpaid($query)
    {
        return $query->where('is_paid', false);
    }

    /**
     * معتمد لكن غير مدفوع
     */
    public function scopeApprovedButUnpaid($query)
    {
        return $query->where('is_approved', true)
            ->where('is_paid', false);
    }

    /**
     * من وافق عليه مستخدم معين
     */
    public function scopeApprovedBy($query, $userId)
    {
        return $query->where('approved_by', $userId);
    }

    /**
     * لجدول عمل معين
     */
    public function scopeForWorkSchedule($query, $workScheduleId)
    {
        return $query->where('work_schedule_id', $workScheduleId);
    }

    /**
     * اليوم
     */
    public function scopeToday($query)
    {
        return $query->whereDate('date', today());
    }

    /**
     * هذا الأسبوع
     */
    public function scopeThisWeek($query)
    {
        return $query->whereBetween('date', [
            now()->startOfWeek(),
            now()->endOfWeek()
        ]);
    }

    /**
     * هذا الشهر
     */
    public function scopeThisMonth($query)
    {
        return $query->whereYear('date', now()->year)
            ->whereMonth('date', now()->month);
    }

    /**
     * أكثر من ساعات معينة
     */
    public function scopeMoreThanHours($query, float $hours)
    {
        $minutes = $hours * 60;
        return $query->where('overtime_minutes', '>=', $minutes);
    }

    /**
     * أقل من ساعات معينة
     */
    public function scopeLessThanHours($query, float $hours)
    {
        $minutes = $hours * 60;
        return $query->where('overtime_minutes', '<=', $minutes);
    }
}
