<?php

namespace App\Models;

use App\Enums\OrderPaymentStatus;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

/**
 * مدفوعات الطلب
 */
class OrderPayment extends Model
{
    protected $fillable = [
        'order_id',
        'method',
        'provider',
        'reference_id',
        'amount',
        'currency_id',
        'status',
        'paid_at',
        'payment_url',
        'webhook_secret',
        'gateway_code',
        'gateway_order_id',
        'webhook_payload',
        'error_message',
    ];

    protected function casts(): array
    {
        return [
            'status' => OrderPaymentStatus::class,
            'amount' => 'integer',
            'paid_at' => 'datetime',
            'webhook_payload' => 'array',
        ];
    }

    /* ================== Boot ================== */

    protected static function booted(): void
    {
        static::creating(function (self $payment) {
            if (empty($payment->reference_id)) {
                $payment->reference_id = static::generateReferenceId();
            }
        });

        static::updated(function (self $payment) {
            // تحديث حالة الدفع في الطلب
            if ($payment->isDirty('status')) {
                $payment->updateOrderPaymentStatus();
            }
        });
    }

    /* ================== العلاقات ================== */

    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    /* ================== Scopes ================== */

    public function scopePaid($query)
    {
        return $query->where('status', OrderPaymentStatus::Paid);
    }

    public function scopePending($query)
    {
        return $query->whereIn('status', [
            OrderPaymentStatus::Initiated,
            OrderPaymentStatus::Pending,
        ]);
    }

    public function scopeFailed($query)
    {
        return $query->where('status', OrderPaymentStatus::Failed);
    }

    /* ================== Accessors ================== */

    /**
     * هل الدفع ناجح؟
     */
    public function getIsSuccessfulAttribute(): bool
    {
        return $this->status === OrderPaymentStatus::Paid;
    }

    /**
     * هل الدفع في حالة نهائية؟
     */
    public function getIsFinalAttribute(): bool
    {
        return $this->status->isFinal();
    }

    /* ================== Methods ================== */

    /**
     * توليد رقم مرجعي فريد
     */
    public static function generateReferenceId(): string
    {
        do {
            $reference = 'PAY-'.strtoupper(Str::random(12));
        } while (static::where('reference_id', $reference)->exists());

        return $reference;
    }

    /**
     * تحديث حالة الدفع
     */
    public function markAsPaid(): void
    {
        $this->update([
            'status' => OrderPaymentStatus::Paid,
            'paid_at' => now(),
        ]);
    }

    /**
     * تحديث كفاشل
     */
    public function markAsFailed(?string $errorMessage = null): void
    {
        $this->update([
            'status' => OrderPaymentStatus::Failed,
            'error_message' => $errorMessage,
        ]);
    }

    /**
     * تحديث حالة الدفع في الطلب
     */
    protected function updateOrderPaymentStatus(): void
    {
        $order = $this->order;
        $totalPaid = $order->payments()->paid()->sum('amount');

        if ($totalPaid >= $order->grand_total) {
            $order->update(['payment_status' => 'paid']);
        } elseif ($totalPaid > 0) {
            $order->update(['payment_status' => 'partial']);
        } else {
            $order->update(['payment_status' => 'unpaid']);
        }
    }
}
