<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * عنصر الطلب
 */
class OrderItem extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'product_id',
        'product_variant_id',
        'product_name',
        'variant_name',
        'sku',
        'quantity',
        'unit_price',
        'unit_cost',
        'discount_amount',
        'line_total',
        'is_free_item',
        'promotion_id',
    ];

    protected function casts(): array
    {
        return [
            'quantity' => 'integer',
            'unit_price' => 'integer',
            'unit_cost' => 'integer',
            'discount_amount' => 'integer',
            'line_total' => 'integer',
            'is_free_item' => 'boolean',
        ];
    }

    /* ================== Boot ================== */

    protected static function booted(): void
    {
        static::saving(function (self $item) {
            // حساب إجمالي السطر تلقائياً
            $item->line_total = ($item->quantity * $item->unit_price) - $item->discount_amount;
        });
    }

    /* ================== العلاقات ================== */

    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function variant(): BelongsTo
    {
        return $this->belongsTo(ProductVariant::class, 'product_variant_id');
    }

    public function promotion(): BelongsTo
    {
        return $this->belongsTo(Promotion::class);
    }

    /* ================== Accessors ================== */

    /**
     * الربح على هذا العنصر
     */
    public function getProfitAttribute(): int
    {
        if ($this->is_free_item) {
            return -($this->unit_cost * $this->quantity);
        }

        return $this->line_total - ($this->unit_cost * $this->quantity);
    }

    /**
     * نسبة الربح
     */
    public function getProfitMarginAttribute(): float
    {
        if ($this->line_total === 0) {
            return 0;
        }

        return round(($this->profit / $this->line_total) * 100, 2);
    }
}
