<?php

namespace App\Models;

use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Enums\ShippingMethod;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

/**
 * الطلب
 */
class Order extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_number',
        'branch_id',
        'client_id',
        'currency_id',
        'exchange_rate',
        'shipping_address_id',
        'shipping_method',
        'shipping_company_id',
        'subtotal',
        'discount_total',
        'shipping_fee',
        'cod_fee',
        'grand_total',
        'paid_amount',
        'status',
        'payment_status',
        'payment_method',
        'promotion_id',
        'coupon_id',
        'client_discount_amount',
        'customer_notes',
        'internal_notes',
        'confirmed_at',
        'completed_at',
        'canceled_at',
    ];

    protected function casts(): array
    {
        return [
            'status' => OrderStatus::class,
            'payment_status' => PaymentStatus::class,
            'shipping_method' => ShippingMethod::class,
            'exchange_rate' => 'integer',
            'subtotal' => 'integer',
            'discount_total' => 'integer',
            'shipping_fee' => 'integer',
            'cod_fee' => 'integer',
            'grand_total' => 'integer',
            'paid_amount' => 'integer',
            'client_discount_amount' => 'integer',
            'confirmed_at' => 'datetime',
            'completed_at' => 'datetime',
            'canceled_at' => 'datetime',
        ];
    }

    /* ================== Boot ================== */

    protected static function booted(): void
    {
        static::creating(function (self $order) {
            if (empty($order->order_number)) {
                $order->order_number = static::generateOrderNumber($order->branch_id);
            }
        });

        static::created(function (self $order) {
            // تسجيل الحالة الأولى
            $order->statusHistory()->create([
                'old_status' => null,
                'new_status' => $order->status->value,
                'changed_at' => now(),
            ]);
        });

        static::updating(function (self $order) {
            // تسجيل تغيير الحالة
            if ($order->isDirty('status')) {
                $order->statusHistory()->create([
                    'old_status' => $order->getOriginal('status'),
                    'new_status' => $order->status->value,
                    'changed_at' => now(),
                ]);
            }
        });
    }

    /* ================== العلاقات ================== */

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    public function shippingAddress(): BelongsTo
    {
        return $this->belongsTo(ShippingAddress::class);
    }

    public function shippingCompany(): BelongsTo
    {
        return $this->belongsTo(ShippingCompany::class);
    }

    public function promotion(): BelongsTo
    {
        return $this->belongsTo(Promotion::class);
    }

    public function coupon(): BelongsTo
    {
        return $this->belongsTo(Coupon::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    public function payments(): HasMany
    {
        return $this->hasMany(OrderPayment::class);
    }

    public function statusHistory(): HasMany
    {
        return $this->hasMany(OrderStatusHistory::class);
    }

    public function shipment(): HasOne
    {
        return $this->hasOne(Shipment::class);
    }

    public function couponRedemption(): HasOne
    {
        return $this->hasOne(CouponRedemption::class);
    }

    public function invoice(): HasOne
    {
        return $this->hasOne(Invoice::class);
    }

    /* ================== Scopes ================== */

    public function scopePending($query)
    {
        return $query->where('status', OrderStatus::Pending);
    }

    public function scopeActive($query)
    {
        return $query->whereIn('status', [
            OrderStatus::Pending,
            OrderStatus::Confirmed,
            OrderStatus::Processing,
            OrderStatus::ReadyToShip,
            OrderStatus::Shipped,
        ]);
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', OrderStatus::Completed);
    }

    public function scopeCanceled($query)
    {
        return $query->where('status', OrderStatus::Canceled);
    }

    public function scopeUnpaid($query)
    {
        return $query->where('payment_status', PaymentStatus::UNPAID);
    }

    public function scopePaid($query)
    {
        return $query->where('payment_status', PaymentStatus::PAID);
    }

    /* ================== Accessors ================== */

    /**
     * عدد العناصر
     */
    public function getItemsCountAttribute(): int
    {
        return $this->items()->sum('quantity');
    }

    /**
     * هل يمكن إلغاء الطلب؟
     */
    public function getIsCancelableAttribute(): bool
    {
        return $this->status->isCancelable();
    }

    /**
     * المبلغ المدفوع
     */
    public function getPaidAmountAttribute(): int
    {
        return $this->payments()
            ->where('status', 'paid')
            ->sum('amount');
    }

    /**
     * المبلغ المتبقي
     */
    public function getRemainingAmountAttribute(): int
    {
        return max(0, $this->grand_total - $this->paid_amount);
    }

    /* ================== Methods ================== */

    /**
     * توليد رقم الطلب
     */
    public static function generateOrderNumber(int $branchId): string
    {
        $counter = OrderCounter::firstOrCreate(
            ['branch_id' => $branchId],
            ['current_sequence' => 0]
        );

        $counter->increment('current_sequence');

        $prefix = 'ORD';
        $branchPart = str_pad((string) $branchId, 2, '0', STR_PAD_LEFT);
        $datePart = now()->format('ymd');
        $seqPart = str_pad((string) $counter->current_sequence, 5, '0', STR_PAD_LEFT);

        return "{$prefix}-{$branchPart}-{$datePart}-{$seqPart}";
    }

    /**
     * تأكيد الطلب
     */
    public function confirm(): bool
    {
        if ($this->status !== OrderStatus::Pending) {
            return false;
        }

        $this->update([
            'status' => OrderStatus::Confirmed,
            'confirmed_at' => now(),
        ]);

        return true;
    }

    /**
     * إلغاء الطلب
     */
    public function cancel(?string $reason = null): bool
    {
        if (! $this->is_cancelable) {
            return false;
        }

        $this->update([
            'status' => OrderStatus::Canceled,
            'canceled_at' => now(),
            'internal_notes' => $reason
                ? ($this->internal_notes ? $this->internal_notes."\n" : '')."سبب الإلغاء: {$reason}"
                : $this->internal_notes,
        ]);

        return true;
    }

    /**
     * إكمال الطلب
     */
    public function complete(): bool
    {
        if ($this->status !== OrderStatus::Delivered) {
            return false;
        }

        $this->update([
            'status' => OrderStatus::Completed,
            'completed_at' => now(),
        ]);

        return true;
    }
}
