<?php

declare(strict_types=1);

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class InvoiceItem extends Model
{
    public const TYPE_PRODUCT = 'product';

    public const TYPE_ACCESSORY = 'accessory';

    protected $table = 'invoice_items';

    protected $fillable = [
        'invoice_id',

        'item_type',

        'product_id',
        'product_variant_id',

        'unit_price',    // int
        'quantity',      // int
        'line_total',    // int

        'notes',
    ];

    protected $casts = [
        'quantity' => 'integer',
        'unit_price' => 'integer',
        'line_total' => 'integer',
    ];

    /* =========================
     * العلاقات
     * ========================= */

    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function productVariant(): BelongsTo
    {
        return $this->belongsTo(ProductVariant::class);
    }

    /* =========================
     * Helpers
     * ========================= */

    public function isProduct(): bool
    {
        return $this->item_type === self::TYPE_PRODUCT;
    }

    public function isAccessory(): bool
    {
        return $this->item_type === self::TYPE_ACCESSORY;
    }

    /**
     * حساب إجمالي السطر كعدد صحيح:
     * unit_price * quantity
     */
    public function recalcLineTotal(): int
    {
        $unitPrice = max(0, (int) ($this->unit_price ?? 0));
        $qty = max(1, (int) ($this->quantity ?? 1));

        return max(0, $unitPrice * $qty);
    }

    protected static function booted(): void
    {
        static::saving(function (self $item): void {
            $item->unit_price = max(0, (int) ($item->unit_price ?? 0));
            $item->quantity = max(1, (int) ($item->quantity ?? 1));

            $item->line_total = $item->recalcLineTotal();
        });

        // بعد حفظ البند: تحديث إجماليات الفاتورة (أعداد صحيحة) ثم تحديث العداد عبر Invoice::created (إن كانت فاتورة جديدة)
        static::saved(function (self $item): void {
            if ($item->invoice) {
                $item->invoice->updateTotals();
            }
        });

        static::deleted(function (self $item): void {
            if ($item->invoice) {
                $item->invoice->updateTotals();
            }
        });
    }
}
