<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class InvoiceCounter extends Model
{
    protected $table = 'invoice_counters';

    protected $fillable = [
        'branch_id',
        'current_sequence',
    ];

    protected $casts = [
        'branch_id' => 'integer',
        'current_sequence' => 'integer',
    ];

    /* =========================
     * Helpers
     * ========================= */

    public static function maxIssuedSequenceForBranch(int $branchId): int
    {
        if ($branchId <= 0) {
            return 0;
        }

        // invoice_number عندك string (000001)، نجيب أكبر رقم كـ UNSIGNED
        $max = (int) DB::table('invoices')
            ->where('branch_id', $branchId)
            ->selectRaw('COALESCE(MAX(CAST(invoice_number AS UNSIGNED)), 0) AS mx')
            ->value('mx');

        return max(0, $max);
    }

    public static function peekNextFormatted(int $branchId, int $pad = 6): string
    {
        if ($branchId <= 0) {
            return str_pad('1', $pad, '0', STR_PAD_LEFT);
        }

        $counterSeq = (int) self::query()
            ->where('branch_id', $branchId)
            ->value('current_sequence');

        $maxIssued = self::maxIssuedSequenceForBranch($branchId);

        $next = max($counterSeq, $maxIssued) + 1;

        return str_pad((string) $next, $pad, '0', STR_PAD_LEFT);
    }

    /**
     * ✅ حجز رقم جديد بطريقة ذرّية (Atomic) + Sync تلقائي مع invoices
     * يمنع التكرار حتى مع إضافة بنفس الوقت، وحتى إذا العداد متأخر.
     */
    public static function reserveNextFormatted(int $branchId, int $pad = 6): string
    {
        if ($branchId <= 0) {
            throw new \InvalidArgumentException('branch_id غير صالح.');
        }

        return DB::transaction(function () use ($branchId, $pad) {

            // 1) حاول تجيب العداد بقفل
            $counter = self::query()
                ->where('branch_id', $branchId)
                ->lockForUpdate()
                ->first();

            // 2) إذا غير موجود: أنشئه أولاً على maxIssued ثم أعد جلبه بقفل
            if (! $counter) {
                $maxIssued = self::maxIssuedSequenceForBranch($branchId);

                $created = self::query()->create([
                    'branch_id' => $branchId,
                    'current_sequence' => $maxIssued,
                ]);

                $counter = self::query()
                    ->whereKey($created->getKey())
                    ->lockForUpdate()
                    ->first();
            }

            // 3) Sync: إذا العداد متأخر عن invoices، عدّله
            $maxIssued = self::maxIssuedSequenceForBranch($branchId);

            if ($maxIssued > (int) $counter->current_sequence) {
                $counter->current_sequence = $maxIssued;
            }

            // 4) increment + save
            $counter->current_sequence = (int) $counter->current_sequence + 1;
            $counter->save();

            return str_pad((string) $counter->current_sequence, $pad, '0', STR_PAD_LEFT);
        }, 3);
    }

    /**
     * اختياري: مزامنة يدوية
     */
    public static function syncCurrentToMaxForBranch(int $branchId): int
    {
        if ($branchId <= 0) {
            return 0;
        }

        return DB::transaction(function () use ($branchId) {

            $maxIssued = self::maxIssuedSequenceForBranch($branchId);

            $counter = self::query()
                ->where('branch_id', $branchId)
                ->lockForUpdate()
                ->first();

            if (! $counter) {
                $counter = self::query()->create([
                    'branch_id' => $branchId,
                    'current_sequence' => $maxIssued,
                ]);
            } else {
                $counter->current_sequence = max((int) $counter->current_sequence, $maxIssued);
                $counter->save();
            }

            return (int) $counter->current_sequence;
        }, 3);
    }
}
