<?php

namespace App\Models;

use App\Models\Concerns\HasBranch;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Invoice extends Model
{
    use HasBranch;
    use SoftDeletes;

    protected $table = 'invoices';

    /**
     * ملاحظة:
     * invoice_number محفوظ كسلسلة بطول 6 (مثال 000001).
     * يفضّل أن يكون لديك unique مركّب (branch_id, invoice_number) بالمهاجرة.
     */
    protected $fillable = [
        'branch_id',
        'client_id',
        'order_id',
        'currency_id',
        'exchange_rate',

        'invoice_number',   // 000001
        'invoice_type',     // ready ✅
        'sale_type',        // retail / wholesale / agent ...
        'status',           // draft / approved / canceled ...
        'payment_status',   // unpaid / partial / paid / installment ...
        'invoice_date',
        'due_date',
        'subtotal',
        'discount_total',
        'shipping_total',
        'grand_total',
        'paid_amount',
        'due_amount',
        'notes',
        'internal_notes',
        'user_id',
        'approved_by',
        'approved_at',
    ];

    protected $casts = [
        'branch_id' => 'integer',
        'client_id' => 'integer',
        'order_id' => 'integer',
        'currency_id' => 'integer',
        'exchange_rate' => 'integer',
        'user_id' => 'integer',
        'approved_by' => 'integer',

        'invoice_date' => 'date',
        'due_date' => 'date',
        'approved_at' => 'datetime',

        // ✅ النوع
        'invoice_type' => 'string',

        // مبالغ كأعداد صحيحة
        'subtotal' => 'integer',
        'discount_total' => 'integer',
        'shipping_total' => 'integer',
        'grand_total' => 'integer',
        'paid_amount' => 'integer',
        'due_amount' => 'integer',
    ];

    /* =========================
     * العلاقات
     * ========================= */

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(InvoiceItem::class);
    }

    public function stockMovements(): MorphMany
    {
        return $this->morphMany(StockMovement::class, 'source');
    }

    /* =========================
     * Helpers (أعداد صحيحة دائمًا)
     * ========================= */

    public function getDiscountTotalAttribute($value): int
    {
        return max(0, (int) ($value ?? 0));
    }

    public function getShippingTotalAttribute($value): int
    {
        return max(0, (int) ($value ?? 0));
    }

    public function getPaidAmountAttribute($value): int
    {
        return max(0, (int) ($value ?? 0));
    }

    public function getDueAmountAttribute($value): int
    {
        return max(0, (int) ($value ?? 0));
    }

    /* =========================
     * Business Logic
     * ========================= */

    public function isReady(): bool
    {
        return true;
    }

    public function updateTotals(): void
    {
        $subtotal = (int) $this->items()->sum('line_total');

        $discount = max(0, (int) ($this->discount_total ?? 0));
        $shipping = max(0, (int) ($this->shipping_total ?? 0));
        $paid = max(0, (int) ($this->paid_amount ?? 0));

        $grand = max(0, (int) ($subtotal - $discount + $shipping));
        $due = max(0, (int) ($grand - $paid));

        $this->forceFill([
            'subtotal' => $subtotal,
            'grand_total' => $grand,
            'due_amount' => $due,
            'discount_total' => $discount,
            'shipping_total' => $shipping,
            'paid_amount' => $paid,
        ])->saveQuietly();
    }

    public function deleteStockMovements(): void
    {
        $this->stockMovements()->delete();
    }

    /* =========================
     * Scopes
     * ========================= */

    public function scopeForBranch(Builder $query, int $branchId): Builder
    {
        return $query->where('branch_id', $branchId);
    }

    public function scopeReady(Builder $query): Builder
    {
        return $query->where('invoice_type', 'ready');
    }

    /* =========================
     * Events
     * ========================= */

    protected static function booted(): void
    {
        static::saving(function (self $invoice): void {
            // ✅ تثبيت invoice_type
            $invoice->invoice_type = 'ready';

            // تثبيت invoice_number كـ 6 خانات
            if ($invoice->invoice_number !== null && $invoice->invoice_number !== '') {
                $digits = preg_replace('/\D/', '', (string) $invoice->invoice_number) ?: '';
                if ($digits !== '') {
                    $invoice->invoice_number = str_pad($digits, 6, '0', STR_PAD_LEFT);
                }
            }

            // تثبيت المبالغ كأعداد صحيحة + منع السالب
            foreach (
                [
                    'subtotal',
                    'discount_total',
                    'shipping_total',
                    'grand_total',
                    'paid_amount',
                    'due_amount',
                ] as $field
            ) {
                $invoice->{$field} = max(0, (int) ($invoice->{$field} ?? 0));
            }
        });
    }
}
