<?php

namespace App\Models;

use App\Models\Concerns\HasBranch;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Expense extends Model
{
    use HasFactory, SoftDeletes, HasBranch;

    protected $fillable = [
        'branch_id',
        'expense_category_id',
        'user_id',
        'amount',
        'description',
        'spent_at',
    ];

    protected $casts = [
        'amount'   => 'integer',   // مبالغ أعداد صحيحة
        'spent_at' => 'datetime',
    ];

    /* ==================== العلاقات ==================== */

    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    public function category()
    {
        return $this->belongsTo(ExpenseCategory::class, 'expense_category_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /* ==================== سكوبات مفيدة ==================== */

    // حسب الفرع
    public function scopeForBranch($query, $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    // حسب المدى الزمني
    public function scopeBetweenDates($query, ?string $from, ?string $to)
    {
        if ($from && $to) {
            return $query->whereBetween('spent_at', [$from, $to]);
        }
        if ($from) {
            return $query->where('spent_at', '>=', $from);
        }
        if ($to) {
            return $query->where('spent_at', '<=', $to);
        }
        return $query;
    }

    // حسب الفئة
    public function scopeForCategory($query, $categoryId)
    {
        return $query->where('expense_category_id', $categoryId);
    }
}
