<?php

namespace App\Models;

use App\Models\Concerns\HasBranch;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class EmployeeAdvance extends Model
{
    use HasFactory, SoftDeletes, HasBranch;

    protected $table = 'employee_advances';

    protected $fillable = [
        'employee_id',
        'branch_id',
        'user_id',
        'total_amount',
        'remaining_amount',
        'deduction_type',
        'deduction_value',
        'start_year',
        'start_month',
        'is_active',
        'notes',
    ];

    protected $casts = [
        'employee_id'      => 'integer',
        'branch_id'        => 'integer',
        'user_id'          => 'integer',
        'total_amount'     => 'integer',
        'remaining_amount' => 'integer',
        'deduction_value'  => 'integer',
        'start_year'       => 'integer',
        'start_month'      => 'integer',
        'is_active'        => 'boolean',
    ];

    /* =========================
     * العلاقات
     * ========================= */

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * أقساط/حركات خصم السلفة (advance_installments.advance_id)
     */
    public function installments(): HasMany
    {
        return $this->hasMany(AdvanceInstallment::class, 'advance_id');
    }

    /* =========================
     * Scopes
     * ========================= */

    public function scopeActive(Builder $query): Builder
    {
        return $query->where('is_active', true)->where('remaining_amount', '>', 0);
    }

    public function scopeApplicableFor(Builder $query, int $year, int $month): Builder
    {
        return $query->active()->where(function (Builder $q) use ($year, $month) {
            $q->where('start_year', '<', $year)
                ->orWhere(function (Builder $qq) use ($year, $month) {
                    $qq->where('start_year', $year)
                        ->where('start_month', '<=', $month);
                });
        });
    }

    /* =========================
     * Accessors
     * ========================= */

    public function getLabelAttribute(): string
    {
        $employee = $this->employee?->name ?? '—';
        $period = sprintf('%04d/%02d', (int) $this->start_year, (int) $this->start_month);

        return $employee . ' - ' . $period;
    }

    /* =========================
     * Helpers
     * ========================= */

    /**
     * هل السلفة فعالة لهذا الشهر؟
     */
    public function isApplicableFor(int $year, int $month): bool
    {
        if (! $this->is_active || (int) $this->remaining_amount <= 0) {
            return false;
        }

        return ($year > (int) $this->start_year)
            || ($year === (int) $this->start_year && $month >= (int) $this->start_month);
    }

    /**
     * حساب مبلغ الخصم لهذا الشهر (لا يتجاوز remaining_amount)
     */
    public function calculateDeduction(int $baseSalary): int
    {
        $baseSalary = max(0, (int) $baseSalary);

        if (! $this->is_active || (int) $this->remaining_amount <= 0) {
            return 0;
        }

        $remaining = (int) $this->remaining_amount;

        $amount = match ((string) $this->deduction_type) {
            'full'    => $remaining,
            'fixed'   => (int) ($this->deduction_value ?? 0),
            'percent' => (int) round($baseSalary * (((int) ($this->deduction_value ?? 0)) / 100)),
            default   => 0,
        };

        $amount = max(0, $amount);

        return min($amount, $remaining);
    }

    /**
     * تطبيق الخصم وتحديث المتبقي
     */
    public function applyDeduction(int $amount): void
    {
        $amount = max(0, (int) $amount);

        if ($amount === 0 || (int) $this->remaining_amount <= 0) {
            return;
        }

        $this->remaining_amount = max(0, (int) $this->remaining_amount - $amount);

        if ((int) $this->remaining_amount === 0) {
            $this->is_active = false;
        }

        $this->save();
    }
}
