<?php

namespace App\Models;

use Spatie\Image\Enums\Fit;
use Spatie\MediaLibrary\HasMedia;
use App\Models\Concerns\HasBranch;
use Illuminate\Database\Eloquent\Model;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Employee extends Model implements HasMedia
{
    use HasFactory, HasBranch, InteractsWithMedia;

    public const SALARY_MONTHLY = 'monthly';
    public const SALARY_DAILY   = 'daily';

    public const GENDER_MALE   = 'male';
    public const GENDER_FEMALE = 'female';

    protected $fillable = [
        'branch_id',
        'pay_grade_id',
        'name',
        'phone',
        'job_title',
        'address',
        'gender',
        'hired_at',
        'salary_type',
        'is_active',
    ];

    protected $casts = [
        'branch_id'    => 'integer',
        'pay_grade_id' => 'integer',
        'is_active'    => 'boolean',
        'hired_at'     => 'date',
    ];

    /* =========================
     * العلاقات
     * ========================= */

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function payGrade(): BelongsTo
    {
        return $this->belongsTo(PayGrade::class);
    }

    /** كشوف الرواتب لهذا الموظف */
    public function payrolls(): HasMany
    {
        return $this->hasMany(Payroll::class);
    }

    public function attendances(): HasMany
    {
        return $this->hasMany(Attendance::class);
    }

    public function absences(): HasMany
    {
        return $this->hasMany(Absence::class);
    }

    public function workSchedules(): HasMany
    {
        return $this->hasMany(WorkSchedule::class);
    }

    public function overtimeRecords(): HasMany
    {
        return $this->hasMany(OvertimeRecord::class);
    }

    /* =========================
     * Scopes
     * ========================= */

    public function scopeActive($q)
    {
        return $q->where('is_active', true);
    }

    public function scopeMonthly($q)
    {
        return $q->where('salary_type', self::SALARY_MONTHLY);
    }

    public function scopeDaily($q)
    {
        return $q->where('salary_type', self::SALARY_DAILY);
    }

    /* =========================
     * Helpers
     * ========================= */

    public function isMonthly(): bool
    {
        return (string) $this->salary_type === self::SALARY_MONTHLY;
    }

    public function isDaily(): bool
    {
        return (string) $this->salary_type === self::SALARY_DAILY;
    }

    /**
     * الراتب الأساسي الشهري (INT)
     * - للموظف الشهري: من الدرجة الوظيفية (PayGrade->amount)
     * - لليومي: يرجع 0 (يُحسب من نظام الحضور/الأيام إن وجد)
     */
    public function getBasicMonthlySalaryAttribute(): int
    {
        if (! $this->isMonthly()) {
            return 0;
        }

        return (int) ($this->payGrade?->amount ?? 0);
    }

    /**
     * اسم نوع الراتب بالعربي
     */
    public function getSalaryTypeLabelAttribute(): string
    {
        return match ((string) $this->salary_type) {
            self::SALARY_MONTHLY => 'شهري',
            self::SALARY_DAILY   => 'يومي',
            default              => '-',
        };
    }

    /** (اختياري) عرض الجنس بالعربي */
    public function getGenderLabelAttribute(): string
    {
        return match ((string) $this->gender) {
            self::GENDER_MALE   => 'ذكر',
            self::GENDER_FEMALE => 'أنثى',
            default             => '-',
        };
    }

    /* =========================
     * Media Library
     * ========================= */

    /** مجموعة ميديا للصورة الشخصية */
    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('avatar')
            ->singleFile()
            ->useDisk('upload') // غيّرها إذا لازم
            ->acceptsMimeTypes(['image/jpeg', 'image/png', 'image/webp'])
            ->useFallbackUrl(asset('images/svg/avatars/blank.svg'))
            ->useFallbackPath(public_path('images/svg/avatars/blank.svg'));
    }

    /** التحويلات */
    public function registerMediaConversions(?Media $media = null): void
    {
        $this->addMediaConversion('thumb')
            ->fit(Fit::Crop, 128, 128)
            ->performOnCollections('avatar')
            ->format('webp')
            ->nonQueued();

        $this->addMediaConversion('preview')
            ->fit(Fit::Crop, 600, 600)
            ->performOnCollections('avatar')
            ->format('webp')
            ->nonQueued();
    }

    /** رابط الصورة (thumb) */
    public function getAvatarUrlAttribute(): string
    {
        $url = $this->getFirstMediaUrl('avatar', 'thumb');

        return $url !== '' ? $url : asset('images/default-avatar.png');
    }
}
