<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * العملة
 */
class Currency extends Model
{
    protected $fillable = [
        'name',
        'code',
        'symbol',
        'decimal_places',
        'position',
        'is_default',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'decimal_places' => 'integer',
            'is_default' => 'boolean',
            'is_active' => 'boolean',
        ];
    }

    /* ================== Boot ================== */

    protected static function booted(): void
    {
        // ضمان وجود عملة افتراضية واحدة فقط
        static::saving(function (self $currency) {
            if ($currency->is_default) {
                static::where('id', '!=', $currency->id ?? 0)
                    ->update(['is_default' => false]);
            }
        });
    }

    /* ================== العلاقات ================== */

    public function exchangeRates(): HasMany
    {
        return $this->hasMany(ExchangeRate::class);
    }

    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    public function carts(): HasMany
    {
        return $this->hasMany(Cart::class);
    }

    /* ================== Scopes ================== */

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeDefault($query)
    {
        return $query->where('is_default', true);
    }

    /* ================== Accessors ================== */

    /**
     * سعر الصرف الحالي
     */
    public function getCurrentRateAttribute(): ?int
    {
        return $this->exchangeRates()
            ->where('is_active', true)
            ->orderBy('created_at', 'desc')
            ->value('rate');
    }

    /* ================== Methods ================== */

    /**
     * تنسيق المبلغ
     */
    public function format(int $amount): string
    {
        $formatted = number_format($amount, $this->decimal_places);

        return $this->position === 'before'
            ? "{$this->symbol}{$formatted}"
            : "{$formatted} {$this->symbol}";
    }

    /**
     * الحصول على العملة الافتراضية
     */
    public static function getDefault(): ?self
    {
        return static::active()->default()->first();
    }

    /**
     * الحصول على ID العملة الافتراضية
     */
    public static function getDefaultId(): ?int
    {
        return static::active()->default()->value('id');
    }
}
