<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * كوبون الخصم
 */
class Coupon extends Model
{
    use HasFactory;

    protected $fillable = [
        'code',
        'name',
        'description',
        'type',
        'discount_value',
        'min_purchase_amount',
        'max_discount_amount',
        'usage_limit_total',
        'usage_limit_per_client',
        'used_count',
        'starts_at',
        'ends_at',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'discount_value' => 'decimal:2',
            'min_purchase_amount' => 'integer',
            'max_discount_amount' => 'integer',
            'usage_limit_total' => 'integer',
            'usage_limit_per_client' => 'integer',
            'used_count' => 'integer',
            'starts_at' => 'datetime',
            'ends_at' => 'datetime',
            'is_active' => 'boolean',
        ];
    }

    /* ================== العلاقات ================== */

    public function redemptions(): HasMany
    {
        return $this->hasMany(CouponRedemption::class);
    }

    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    /* ================== Scopes ================== */

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeValid($query)
    {
        $now = now();

        return $query->where('is_active', true)
            ->where(function ($q) use ($now) {
                $q->whereNull('starts_at')
                    ->orWhere('starts_at', '<=', $now);
            })
            ->where(function ($q) use ($now) {
                $q->whereNull('ends_at')
                    ->orWhere('ends_at', '>=', $now);
            })
            ->where(function ($q) {
                $q->whereNull('usage_limit_total')
                    ->orWhereColumn('used_count', '<', 'usage_limit_total');
            });
    }

    public function scopeByCode($query, string $code)
    {
        return $query->where('code', strtoupper(trim($code)));
    }

    /* ================== Accessors ================== */

    /**
     * هل الكوبون ساري المفعول؟
     */
    public function getIsValidAttribute(): bool
    {
        if (! $this->is_active) {
            return false;
        }

        $now = now();

        if ($this->starts_at && $this->starts_at->isFuture()) {
            return false;
        }

        if ($this->ends_at && $this->ends_at->isPast()) {
            return false;
        }

        if ($this->usage_limit_total && $this->used_count >= $this->usage_limit_total) {
            return false;
        }

        return true;
    }

    /**
     * الاستخدامات المتبقية
     */
    public function getRemainingUsageAttribute(): ?int
    {
        if ($this->usage_limit_total === null) {
            return null;
        }

        return max(0, $this->usage_limit_total - $this->used_count);
    }

    /* ================== Methods ================== */

    /**
     * هل يمكن للعميل استخدام هذا الكوبون؟
     */
    public function canBeUsedBy(Client $client): bool
    {
        if (! $this->is_valid) {
            return false;
        }

        // التحقق من حد الاستخدام لكل عميل
        if ($this->usage_limit_per_client) {
            $clientUsage = $this->redemptions()
                ->where('client_id', $client->id)
                ->count();

            if ($clientUsage >= $this->usage_limit_per_client) {
                return false;
            }
        }

        return true;
    }

    /**
     * حساب مبلغ الخصم
     */
    public function calculateDiscount(int $amount): int
    {
        if (! $this->is_valid) {
            return 0;
        }

        // التحقق من الحد الأدنى للشراء
        if ($this->min_purchase_amount && $amount < $this->min_purchase_amount) {
            return 0;
        }

        $discount = match ($this->type) {
            'percentage' => (int) round($amount * ($this->discount_value / 100)),
            'fixed' => (int) $this->discount_value,
            default => 0,
        };

        // تطبيق الحد الأقصى للخصم
        if ($this->max_discount_amount && $discount > $this->max_discount_amount) {
            $discount = $this->max_discount_amount;
        }

        return $discount;
    }

    /**
     * تسجيل استخدام الكوبون
     */
    public function redeem(Client $client, Order $order, int $discountAmount): CouponRedemption
    {
        $this->increment('used_count');

        return $this->redemptions()->create([
            'client_id' => $client->id,
            'order_id' => $order->id,
            'discount_amount' => $discountAmount,
            'redeemed_at' => now(),
        ]);
    }

    /**
     * البحث عن كوبون بالكود
     */
    public static function findByCode(string $code): ?self
    {
        return static::byCode($code)->first();
    }
}
