<?php

namespace App\Models\Concerns;

use App\Models\Branch;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

trait HasBranch
{
    protected static bool $branchScopeEnabled = true;
    protected static bool $branchScopeDisabledOnce = false;
    protected static $__branchScopeUserCache = null;

    protected static function bootHasBranch(): void
    {
        // ربط الفرع تلقائيًا عند الإنشاء (لمن لديه branch_id فقط)
        static::creating(function (Model $model) {
            if (! static::shouldApplyBranchScope()) {
                return;
            }

            $user = static::branchUser();
            if ($user && $user->branch_id && ! $model->getAttribute('branch_id')) {
                $model->setAttribute('branch_id', $user->branch_id);
            }
        });

        // فلترة جميع الاستعلامات تلقائيًا
        static::addGlobalScope('branchScope', function (Builder $builder) {
            if (static::shouldApplyBranchScope()) {
                $user  = static::branchUser();
                $table = $builder->getModel()->getTable();
                $builder->where($table . '.branch_id', $user->branch_id);
            }

            if (self::$branchScopeDisabledOnce) {
                self::$branchScopeDisabledOnce = false;
            }
        });
    }

    protected static function shouldApplyBranchScope(): bool
    {
        // إيقاف عام / لمرة واحدة
        if (! self::$branchScopeEnabled || self::$branchScopeDisabledOnce) return false;

        // في الكونسول بدون Auth (migrate/seed/test) → لا تطبق
        if (App::runningInConsole() && ! Auth::check()) return false;

        // حماية: لو طُبّق على موديل User قبل تحميل المستخدم، لا نفلتر
        if (static::class === \App\Models\User::class) {
            $guard = Auth::guard();
            if (method_exists($guard, 'hasUser') && ! $guard->hasUser()) return false;
        }

        $user = static::branchUser();
        if (! $user) return false;

        // ✅ إذا المسؤول branch_id=null → لا فلترة (يرى كل الفروع)
        if (! $user->branch_id) return false;

        // أدوار مستثناة من الفلترة كليًا
        if (method_exists($user, 'hasAnyRole') && $user->hasAnyRole(['admin', 'accounting_manager'])) {
            return false;
        }

        // إذن اختياري لعرض كل الفروع (إن وُجد)
        if (method_exists($user, 'can') && $user->can('branches.view_all')) {
            return false;
        }

        // افتراضيًا: نطبق فلترة الفرع
        return true;
    }

    protected static function branchUser()
    {
        if (self::$__branchScopeUserCache !== null) {
            return self::$__branchScopeUserCache;
        }
        self::$__branchScopeUserCache = Auth::check() ? Auth::user() : null;
        return self::$__branchScopeUserCache;
    }

    // تحكم برمجي
    public static function disableBranchScope(): void
    {
        self::$branchScopeEnabled = false;
    }
    public static function enableBranchScope(): void
    {
        self::$branchScopeEnabled = true;
    }
    public static function disableBranchScopeOnce(): void
    {
        self::$branchScopeDisabledOnce = true;
    }

    // أدوات استعلام
    public static function withoutBranchScope(): Builder
    {
        return static::withoutGlobalScope('branchScope');
    }

    public static function forBranch(int $branchId): Builder
    {
        return static::withoutGlobalScope('branchScope')->where('branch_id', $branchId);
    }

    public static function forBranches(array $branchIds): Builder
    {
        return static::withoutGlobalScope('branchScope')->whereIn('branch_id', $branchIds);
    }

    // العلاقة
    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }
}
