<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * خصم العميل المسجل
 */
class ClientDiscount extends Model
{
    protected $fillable = [
        'client_id',
        'discount_type',
        'discount_value',
        'starts_at',
        'ends_at',
        'is_active',
        'created_by',
        'notes',
    ];

    protected function casts(): array
    {
        return [
            'discount_value' => 'decimal:2',
            'starts_at' => 'datetime',
            'ends_at' => 'datetime',
            'is_active' => 'boolean',
        ];
    }

    /* ================== العلاقات ================== */

    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /* ================== Scopes ================== */

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeValid($query)
    {
        $now = now();

        return $query->where('is_active', true)
            ->where(function ($q) use ($now) {
                $q->whereNull('starts_at')
                    ->orWhere('starts_at', '<=', $now);
            })
            ->where(function ($q) use ($now) {
                $q->whereNull('ends_at')
                    ->orWhere('ends_at', '>=', $now);
            });
    }

    /* ================== Accessors ================== */

    /**
     * هل الخصم ساري المفعول؟
     */
    public function getIsValidAttribute(): bool
    {
        if (! $this->is_active) {
            return false;
        }

        $now = now();

        if ($this->starts_at && $this->starts_at->isFuture()) {
            return false;
        }

        if ($this->ends_at && $this->ends_at->isPast()) {
            return false;
        }

        return true;
    }

    /* ================== Methods ================== */

    /**
     * حساب مبلغ الخصم
     */
    public function calculateDiscount(int $amount): int
    {
        if (! $this->is_valid) {
            return 0;
        }

        return match ($this->discount_type) {
            'percentage' => (int) round($amount * ($this->discount_value / 100)),
            'fixed' => (int) $this->discount_value,
            default => 0,
        };
    }

    /**
     * الحصول على الخصم الفعال للعميل
     */
    public static function getActiveForClient(int $clientId): ?self
    {
        return static::where('client_id', $clientId)
            ->valid()
            ->first();
    }
}
