<?php

namespace App\Models;

use App\Models\Concerns\HasBranch;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Hash;

class Client extends Authenticatable
{
    use HasBranch;
    use Notifiable;

    protected $table = 'clients';

    /* ================== الحقول ================== */

    protected $fillable = [
        'branch_id',
        'city_id',
        'name',
        'company',
        'client_type',
        'phone',
        'phone2',
        'address',
        'password',
        'is_active',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    /* ================== Mutators ================== */

    /**
     * حفظ كلمة المرور كـ hash تلقائيًا
     */
    public function setPasswordAttribute($value): void
    {
        if ($value === null || $value === '') {
            return;
        }

        // إذا كانت بالفعل hash (طول شائع) لا نعيد تشفيرها
        $v = (string) $value;
        $this->attributes['password'] = str_starts_with($v, '$2y$') || str_starts_with($v, '$argon2')
            ? $v
            : Hash::make($v);
    }

    /* ================== العلاقات الأساسية ================== */

    public function city(): BelongsTo
    {
        return $this->belongsTo(City::class);
    }

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class, 'branch_id');
    }

    public function branches(): BelongsToMany
    {
        return $this->belongsToMany(Branch::class, 'branch_client')
            ->withTimestamps();
    }

    /* ================== المبيعات ================== */

    public function invoices(): HasMany
    {
        return $this->hasMany(Invoice::class, 'client_id');
    }

    public function receipts(): HasMany
    {
        return $this->hasMany(ClientReceipt::class, 'client_id');
    }

    public function receiptAllocations(): HasMany
    {
        return $this->hasMany(ReceiptInvoiceAllocation::class, 'client_id');
    }

    public function paymentAllocations(): HasMany
    {
        return $this->hasMany(PaymentAllocation::class, 'client_id');
    }

    /* ================== المحاسبة ================== */

    public function account(): HasMany
    {
        return $this->hasMany(ClientAccount::class, 'client_id');
    }

    public function ledgerEntries(): HasMany
    {
        return $this->hasMany(ClientLedgerEntry::class, 'client_id');
    }

    /* ================== الأقساط ================== */

    public function installmentPlans(): HasMany
    {
        return $this->hasMany(ClientInstallmentPlan::class, 'client_id');
    }

    public function advanceInstallments(): HasMany
    {
        return $this->hasMany(AdvanceInstallment::class, 'client_id');
    }

    /* ================== B2C: الطلبات وعناوين الشحن ================== */

    public function orders(): HasMany
    {
        return $this->hasMany(Order::class, 'client_id');
    }

    public function shippingAddresses(): HasMany
    {
        return $this->hasMany(ShippingAddress::class, 'client_id');
    }

    public function defaultShippingAddress(): HasOne
    {
        return $this->hasOne(ShippingAddress::class, 'client_id')->where('is_default', true);
    }

    /* ================== سلة التسوق ================== */

    public function cart(): HasOne
    {
        return $this->hasOne(Cart::class, 'client_id')->active();
    }

    public function carts(): HasMany
    {
        return $this->hasMany(Cart::class, 'client_id');
    }

    /* ================== الملصقات (للمصممين) ================== */

    public function stickers(): HasMany
    {
        return $this->hasMany(Sticker::class, 'client_id');
    }

    public function approvedStickers(): HasMany
    {
        return $this->hasMany(Sticker::class, 'client_id')->where('status', 'approved');
    }

    /* ================== التصاميم ================== */

    public function designs(): HasMany
    {
        return $this->hasMany(ProductDesign::class, 'client_id');
    }

    /* ================== الخصومات والكوبونات ================== */

    public function discount(): HasOne
    {
        return $this->hasOne(ClientDiscount::class, 'client_id');
    }

    public function activeDiscount(): HasOne
    {
        return $this->hasOne(ClientDiscount::class, 'client_id')->valid();
    }

    public function couponRedemptions(): HasMany
    {
        return $this->hasMany(CouponRedemption::class, 'client_id');
    }

    /* ================== Scopes ================== */

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function isDesigner(): bool
    {
        return $this->client_type === 'designer';
    }

    public function isPersonal(): bool
    {
        return $this->client_type === 'personal';
    }

    public function isCompany(): bool
    {
        return $this->client_type === 'company';
    }

    /* ================== Auth: تسجيل الدخول بالهاتف ================== */

    /**
     * يجعل Laravel يستخدم phone كـ "username" لهذا الموديل عند التحقق.
     * (مفيد لبعض السيناريوهات، لكن سنعتمد guard + controller أدناه)
     */
    public function username(): string
    {
        return 'phone';
    }
}
