<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * سلة التسوق
 */
class Cart extends Model
{
    use HasFactory;

    protected $fillable = [
        'client_id',
        'session_id',
        'currency_id',
        'coupon_code',
        'expires_at',
    ];

    protected function casts(): array
    {
        return [
            'expires_at' => 'datetime',
        ];
    }

    /* ================== العلاقات ================== */

    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(CartItem::class);
    }

    /* ================== Scopes ================== */

    /**
     * السلات غير المنتهية الصلاحية
     */
    public function scopeActive($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('expires_at')
                ->orWhere('expires_at', '>', now());
        });
    }

    /**
     * السلات المنتهية الصلاحية
     */
    public function scopeExpired($query)
    {
        return $query->whereNotNull('expires_at')
            ->where('expires_at', '<=', now());
    }

    /* ================== Accessors ================== */

    /**
     * هل السلة منتهية الصلاحية؟
     */
    public function getIsExpiredAttribute(): bool
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    /**
     * إجمالي عدد العناصر
     */
    public function getItemsCountAttribute(): int
    {
        return $this->items()->sum('quantity');
    }

    /**
     * المجموع الفرعي
     */
    public function getSubtotalAttribute(): int
    {
        return $this->items()->sum(\DB::raw('quantity * unit_price'));
    }

    /* ================== Methods ================== */

    /**
     * إضافة عنصر للسلة
     */
    public function addItem(int $productVariantId, int $quantity = 1, int $unitPrice = 0): CartItem
    {
        $variant = ProductVariant::findOrFail($productVariantId);
        $price = $unitPrice ?: $variant->getPriceForType('retail');

        $existingItem = $this->items()->where('product_variant_id', $productVariantId)->first();

        if ($existingItem) {
            // Update existing item - increment quantity
            $existingItem->increment('quantity', $quantity);
            $existingItem->update(['unit_price' => $price]);

            return $existingItem->fresh();
        }

        // Create new item
        return $this->items()->create([
            'product_variant_id' => $productVariantId,
            'product_id' => $variant->product_id,
            'quantity' => $quantity,
            'unit_price' => $price,
        ]);
    }

    /**
     * تحديث كمية عنصر
     */
    public function updateItemQuantity(int $productVariantId, int $quantity): ?CartItem
    {
        $item = $this->items()->where('product_variant_id', $productVariantId)->first();

        if ($item) {
            if ($quantity <= 0) {
                $item->delete();

                return null;
            }

            $item->update(['quantity' => $quantity]);

            return $item->fresh();
        }

        return null;
    }

    /**
     * إزالة عنصر من السلة
     */
    public function removeItem(int $productVariantId): bool
    {
        return $this->items()->where('product_variant_id', $productVariantId)->delete() > 0;
    }

    /**
     * تفريغ السلة
     */
    public function clear(): void
    {
        $this->items()->delete();
        $this->update(['coupon_code' => null]);
    }
}
