<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Hash;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class BranchSetting extends Model implements HasMedia
{
    use HasFactory, InteractsWithMedia;

    protected $table = 'branch_settings';

    /** Media Collections */
    public const MEDIA_LOGO_LIGHT = 'logo_light';

    public const MEDIA_LOGO_DARK = 'logo_dark';

    public const MEDIA_FAVICON = 'favicon';

    public const MEDIA_PRODUCT_CARD_BG = 'product_card_bg';

    /** لكل Template نخزن صورة Header مستقلة */
    public const MEDIA_INVOICE_HEADER_BG_PREFIX = 'invoice_header_bg_';

    /** Invoice Templates */
    public const INVOICE_TEMPLATES = [
        'template-01',
        'template-02',
        'template-03',
        'template-04',
        'template-05',
        'template-06',
        'template-07',
    ];

    protected $fillable = [
        'branch_id',
        'vault_reset_password',
        'theme_primary',
        'theme_palette',
        'address',
        'phones',
        'socials',
        'invoice_design',
        'product_card_design',
        'invoice_settings',
    ];

    protected $casts = [
        'branch_id' => 'integer',

        'theme_palette' => 'array',
        'phones' => 'array',
        'socials' => 'array',

        'invoice_design' => 'array',
        'product_card_design' => 'array',
        'invoice_settings' => 'array',
    ];

    protected $appends = [
        'logo_light_url',
        'logo_dark_url',
        'favicon_url',
        'invoice_css_vars',
        'product_card_bg_url',

        // رابط صورة هيدر القالب الحالي
        'invoice_header_bg_url',
    ];

    protected $hidden = ['vault_reset_password'];

    public function setVaultResetPassword(string $password): void
    {
        $this->update(['vault_reset_password' => Hash::make($password)]);
    }

    public function checkVaultResetPassword(string $password): bool
    {
        return $this->vault_reset_password
            ? Hash::check($password, $this->vault_reset_password)
            : false;
    }

    /* ==========================
     | العلاقات
     ========================== */
    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    /* ==========================
     | Helpers: Template Keys
     ========================== */
    public static function normalizeInvoiceTemplate(?string $template): string
    {
        $template = (string) ($template ?: 'template-01');

        return in_array($template, self::INVOICE_TEMPLATES, true) ? $template : 'template-01';
    }

    public static function invoiceHeaderBgCollectionKey(string $template): string
    {
        return self::MEDIA_INVOICE_HEADER_BG_PREFIX.self::normalizeInvoiceTemplate($template);
    }

    public function invoiceHeaderBgUrl(?string $template = null): ?string
    {
        $template = self::normalizeInvoiceTemplate($template ?: $this->invoiceTemplateKey());
        $col = self::invoiceHeaderBgCollectionKey($template);

        $url = $this->getFirstMediaUrl($col);

        return $url !== '' ? $url : null;
    }

    /* ==========================
     | Media Library
     ========================== */
    public function registerMediaCollections(): void
    {
        $this->addMediaCollection(self::MEDIA_LOGO_LIGHT)
            ->singleFile()
            ->useDisk('upload')
            ->acceptsMimeTypes(['image/png', 'image/jpeg', 'image/webp', 'image/svg+xml']);

        $this->addMediaCollection(self::MEDIA_LOGO_DARK)
            ->singleFile()
            ->useDisk('upload')
            ->acceptsMimeTypes(['image/png', 'image/jpeg', 'image/webp', 'image/svg+xml']);

        $this->addMediaCollection(self::MEDIA_FAVICON)
            ->singleFile()
            ->useDisk('upload')
            ->acceptsMimeTypes(['image/png', 'image/x-icon', 'image/vnd.microsoft.icon', 'image/svg+xml']);

        // Collections لكل Template للهيدر
        foreach (self::INVOICE_TEMPLATES as $t) {
            $this->addMediaCollection(self::invoiceHeaderBgCollectionKey($t))
                ->singleFile()
                ->useDisk('upload')
                ->acceptsMimeTypes(['image/png', 'image/jpeg', 'image/webp']);
        }

        $this->addMediaCollection(self::MEDIA_PRODUCT_CARD_BG)
            ->singleFile()
            ->useDisk('upload')
            ->acceptsMimeTypes(['image/png', 'image/jpeg', 'image/webp']);
    }

    public function registerMediaConversions(?Media $media = null): void
    {
        if (! $media) {
            return;
        }

        if (in_array($media->mime_type, ['image/svg+xml', 'image/x-icon', 'image/vnd.microsoft.icon'], true)) {
            return;
        }

        $this->addMediaConversion('thumb')
            ->width(200)->height(200)
            ->nonQueued();

        $this->addMediaConversion('preview')
            ->width(600)->height(600)
            ->nonQueued();
    }

    /* ==========================
     | Accessors (Media URLs)
     ========================== */
    public function getLogoLightUrlAttribute(): ?string
    {
        $url = $this->getFirstMediaUrl(self::MEDIA_LOGO_LIGHT);

        return $url !== '' ? $url : null;
    }

    public function getLogoDarkUrlAttribute(): ?string
    {
        $url = $this->getFirstMediaUrl(self::MEDIA_LOGO_DARK);

        return $url !== '' ? $url : null;
    }

    public function getFaviconUrlAttribute(): ?string
    {
        $url = $this->getFirstMediaUrl(self::MEDIA_FAVICON);

        return $url !== '' ? $url : null;
    }

    public function getInvoiceHeaderBgUrlAttribute(): ?string
    {
        return $this->invoiceHeaderBgUrl($this->invoiceTemplateKey());
    }

    public function getProductCardBgUrlAttribute(): ?string
    {
        $url = $this->getFirstMediaUrl(self::MEDIA_PRODUCT_CARD_BG);

        return $url !== '' ? $url : null;
    }

    /* ==========================
     | Invoice Template
     ========================== */
    public function invoiceTemplateKey(): string
    {
        $key = (string) data_get($this->invoice_design, 'template', 'template-01');

        return self::normalizeInvoiceTemplate($key);
    }

    /**
     * دمج إعدادات الفاتورة حسب القالب:
     * - defaults
     * - legacy root (للتوافق القديم)
     * - templates[templateKey] (إعدادات القالب الحالي)
     *
     * ملاحظة: تقدر تمرر $template حتى تجيب تصميم قالب آخر.
     */
    public function invoiceDesignMerged(?string $template = null): array
    {
        $storedRoot = is_array($this->invoice_design) ? $this->invoice_design : [];

        $templateKey = self::normalizeInvoiceTemplate($template ?: $this->invoiceTemplateKey());

        $storedTemplate = data_get($storedRoot, "templates.$templateKey", []);
        $storedTemplate = is_array($storedTemplate) ? $storedTemplate : [];

        // legacy: القديم يبقى شغال (بدون templates)
        $legacy = $storedRoot;
        unset($legacy['templates']);

        $defaults = $this->invoiceDesignDefaults();

        // الدمج النهائي: defaults -> legacy -> template-specific
        $merged = $this->arrayMergeRecursiveDistinct($defaults, is_array($legacy) ? $legacy : []);
        $merged = $this->arrayMergeRecursiveDistinct($merged, $storedTemplate);

        // تأكيد القالب
        $merged['template'] = $templateKey;

        // إجبار أعداد صحيحة
        $this->forceInvoiceDesignInts($merged);

        return $merged;
    }

    protected function invoiceDesignDefaults(): array
    {
        return [
            // القالب الحالي (للقراءة داخل القوالب)
            'template' => 'template-01',

            // مكان تخزين إعدادات كل قالب
            'templates' => [],

            // محاذاة
            'align' => [
                'global' => 'right',

                'header' => [
                    'title' => 'right',
                    'branch' => 'right',
                    'meta' => 'right',
                    'pills' => 'left',
                ],

                'client' => [
                    'label' => 'right',
                    'value' => 'right',
                ],

                'table' => [
                    'head' => 'center',
                    'no' => 'center',
                    'product' => 'right',
                    'variant' => 'right',
                    'unit' => 'center',
                    'qty' => 'center',
                    'total' => 'center',
                ],

                'totals' => [
                    'label' => 'right',
                    'value' => 'left',
                ],

                'notes' => [
                    'text' => 'right',
                ],

                'signatures' => [
                    'titles' => 'center',
                    'line' => 'center',
                ],
            ],

            'header' => [
                // خلفية الهيدر
                'background' => [
                    'type' => 'none', // none | color | gradient | image
                    'color' => '#FFFFFF',

                    'gradient' => [
                        'from' => '#FFFFFF',
                        'to' => '#F3F4F6',
                        'angle' => 135,
                    ],

                    'image' => [
                        'fit' => 'cover',     // cover | contain
                        'position' => 'center',    // center | top | bottom | left | right
                        'repeat' => 'no-repeat',
                        'opacity' => 100,         // 0..100
                    ],
                ],

                // توزيع عناصر الهيدر (يمين/وسط/يسار)
                'slots' => [
                    'right' => ['logo', 'branch_name', 'branch_meta'],
                    'center' => [],
                    'left' => ['invoice_title', 'invoice_number', 'invoice_date'],
                ],

                'title' => [
                    'text' => 'فاتورة مبيعات',
                    'size_px' => 22,
                    'weight' => 900,
                ],

                'branch' => [
                    'name_size_px' => 20,
                    'meta_size_px' => 9,
                ],

                'pills' => [
                    'show' => true,
                    'size_px' => 10,
                ],

                'logo' => [
                    'show' => true,
                    'size_mm' => 26,
                    'radius_px' => 14,
                    'bg' => '#FFFFFF',
                    'border' => [
                        'enabled' => true,
                        'width_px' => 2,
                        'color' => '#E5E7EB',
                    ],
                ],
            ],

            'client' => [
                'show' => true,
                'layout' => 'bar',
                'font_px' => 10,
                'bg' => '#E9ECEF',
                'radius_px' => 999,
                'label_weight' => 900,
                'value_weight' => 700,
                'fields' => [
                    'name' => true,
                    'phone' => true,
                    'address' => true,
                ],
            ],

            'table' => [
                'head_font_px' => 9,
                'font_px' => 9,
                'row_alt' => true,
                'show_variant' => true,
                'show_barcode' => true,
                'columns' => [
                    'no' => true,
                    'product' => true,
                    'variant' => true,
                    'unit_price' => true,
                    'qty' => true,
                    'total' => true,
                ],
            ],

            'totals' => [
                'show_footer_total' => true,
                'currency' => [
                    'show' => true,
                    'text' => 'د.ع',
                ],
                'show_shipping' => true,
                'show_discount' => true,
                'show_paid' => true,
                'show_remaining' => true,
                'show_client_debts' => true,
            ],

            'notes' => [
                'show' => true,
                'mode' => 'repeater',
                'items' => [
                    ['text' => 'يرجى الاحتفاظ بهذه الفاتورة كمرجع في حال الاستبدال أو المراجعة.'],
                    ['text' => 'البضاعة المباعة بحالة سليمة عند التسليم.'],
                ],
                'single_text' => '',
            ],

            'signatures' => [
                'show' => true,
                'receiver' => [
                    'label' => 'المستلم',
                ],
                'employee' => [
                    'label' => 'الموظف',
                ],
            ],
        ];
    }

    protected function forceInvoiceDesignInts(array &$merged): void
    {
        // header
        data_set($merged, 'header.title.size_px', (int) data_get($merged, 'header.title.size_px', 22));
        data_set($merged, 'header.title.weight', (int) data_get($merged, 'header.title.weight', 900));
        data_set($merged, 'header.branch.name_size_px', (int) data_get($merged, 'header.branch.name_size_px', 20));
        data_set($merged, 'header.branch.meta_size_px', (int) data_get($merged, 'header.branch.meta_size_px', 9));
        data_set($merged, 'header.logo.size_mm', (int) data_get($merged, 'header.logo.size_mm', 26));
        data_set($merged, 'header.logo.radius_px', (int) data_get($merged, 'header.logo.radius_px', 14));
        data_set($merged, 'header.logo.border.width_px', (int) data_get($merged, 'header.logo.border.width_px', 2));
        data_set($merged, 'header.pills.size_px', (int) data_get($merged, 'header.pills.size_px', 10));

        // header background
        data_set($merged, 'header.background.gradient.angle', (int) data_get($merged, 'header.background.gradient.angle', 135));
        data_set($merged, 'header.background.image.opacity', (int) data_get($merged, 'header.background.image.opacity', 100));

        // client
        data_set($merged, 'client.font_px', (int) data_get($merged, 'client.font_px', 10));
        data_set($merged, 'client.radius_px', (int) data_get($merged, 'client.radius_px', 999));
        data_set($merged, 'client.label_weight', (int) data_get($merged, 'client.label_weight', 900));
        data_set($merged, 'client.value_weight', (int) data_get($merged, 'client.value_weight', 700));

        // table
        data_set($merged, 'table.head_font_px', (int) data_get($merged, 'table.head_font_px', 9));
        data_set($merged, 'table.font_px', (int) data_get($merged, 'table.font_px', 9));
    }

    private function arrayMergeRecursiveDistinct(array $defaults, array $overrides): array
    {
        foreach ($overrides as $key => $value) {
            if (is_array($value) && isset($defaults[$key]) && is_array($defaults[$key])) {
                $defaults[$key] = $this->arrayMergeRecursiveDistinct($defaults[$key], $value);
            } else {
                $defaults[$key] = $value;
            }
        }

        return $defaults;
    }

    /* ==========================
     | Palette + invoice_css_vars
     ========================== */
    public function themePaletteOrGenerate(): array
    {
        $base = $this->normalizedHex($this->theme_primary ?? '#006837');
        $pal = $this->theme_palette;

        if (is_array($pal) && ($pal['base'] ?? null) === $base) {
            return $pal;
        }

        return $this->generatePaletteFromBase($base);
    }

    protected function generatePaletteFromBase(string $hex): array
    {
        $hex = $this->normalizedHex($hex);

        $mid = $this->tint($hex, 12);
        $dark = $this->shade($hex, 15);

        $border = $this->mix($hex, '#DDE6E1', 20);
        $pill = $this->mix('#F3F5F4', $hex, 8);

        return [
            'base' => $hex,
            'mid' => $mid,
            'dark' => $dark,

            'muted' => '#7E8B86',
            'field' => '#E9ECEF',
            'bg' => '#FFFFFF',
            'pill' => $pill,
            'border' => $border,
            'ink' => '#0A3326',
        ];
    }

    public function getInvoiceCssVarsAttribute(): array
    {
        $palette = $this->themePaletteOrGenerate();

        return [
            '--primary' => $palette['base'] ?? '#006837',
            '--primary-400' => $palette['mid'] ?? '#1F7A52',
            '--primary-700' => $palette['dark'] ?? '#00512B',
            '--text' => $palette['ink'] ?? '#0A3326',
            '--text-muted' => $palette['muted'] ?? '#7E8B86',
            '--surface' => $palette['bg'] ?? '#FFFFFF',
            '--surface-field' => $palette['field'] ?? '#E9ECEF',
            '--surface-pill' => $palette['pill'] ?? '#F3F5F4',
            '--border' => $palette['border'] ?? '#DDE6E1',
        ];
    }

    /* ==========================
     | Color Helpers
     ========================== */
    protected function normalizedHex(?string $hex): string
    {
        $hex = $hex ? trim($hex) : '#006837';

        if (! str_starts_with($hex, '#')) {
            $hex = '#'.$hex;
        }

        if (strlen($hex) === 4) {
            $hex = '#'.$hex[1].$hex[1].$hex[2].$hex[2].$hex[3].$hex[3];
        }

        if (strlen($hex) !== 7) {
            $hex = '#006837';
        }

        return strtoupper($hex);
    }

    protected function hexToRgb(string $hex): array
    {
        $hex = ltrim($this->normalizedHex($hex), '#');

        return [
            'r' => hexdec(substr($hex, 0, 2)),
            'g' => hexdec(substr($hex, 2, 2)),
            'b' => hexdec(substr($hex, 4, 2)),
        ];
    }

    protected function rgbToHex(int $r, int $g, int $b): string
    {
        $r = max(0, min(255, $r));
        $g = max(0, min(255, $g));
        $b = max(0, min(255, $b));

        return sprintf('#%02X%02X%02X', $r, $g, $b);
    }

    protected function mix(string $colorA, string $colorB, int $weight): string
    {
        $weight = max(0, min(100, $weight));

        $a = $this->hexToRgb($colorA);
        $b = $this->hexToRgb($colorB);

        $r = (int) round($a['r'] + (($b['r'] - $a['r']) * ($weight / 100)));
        $g = (int) round($a['g'] + (($b['g'] - $a['g']) * ($weight / 100)));
        $bl = (int) round($a['b'] + (($b['b'] - $a['b']) * ($weight / 100)));

        return $this->rgbToHex($r, $g, $bl);
    }

    protected function tint(string $hex, int $percent): string
    {
        $percent = max(0, min(100, $percent));

        return $this->mix($hex, '#FFFFFF', $percent);
    }

    protected function shade(string $hex, int $percent): string
    {
        $percent = max(0, min(100, $percent));

        return $this->mix($hex, '#000000', $percent);
    }
}
