<?php

namespace App\Models;

// ✅ هذا هو جدول product_prices (بدون branch_id)

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class Branch extends Model
{
    use HasFactory;

    public const STATUS_ACTIVE = true;

    public const STATUS_INACTIVE = false;

    protected $fillable = [
        'name',
        'city_id',
        'status',
    ];

    protected $casts = [
        'status' => 'boolean',
    ];

    protected $with = [
        'settings',
    ];

    protected $appends = [
        'status_name',
        'logo_light_url',
        'logo_dark_url',
        'favicon_url',
        'invoice_css_vars',
    ];

    /* ================== Hooks ================== */

    protected static function booted(): void
    {
        static::deleting(function (Branch $branch) {

            DB::transaction(function () use ($branch) {

                // =========================================================
                // Pivot
                // =========================================================
                $branch->warehouses()->detach();
                $branch->clients()->detach();

                // =========================================================
                // HasOne
                // =========================================================
                $branch->settings()?->delete();

                // =========================================================
                // Vault (vault_transactions عبر vault_id)
                // =========================================================
                $branch->vaultTransactions()->delete();
                $branch->vaults()->delete();

                // =========================================================
                // Receipts allocations (receipt_invoice_allocations عبر client_receipt_id)
                // =========================================================
                $branch->receiptInvoiceAllocations()->delete();

                if (in_array(\Illuminate\Database\Eloquent\SoftDeletes::class, class_uses_recursive(ClientReceipt::class), true)) {
                    $branch->clientReceipts()->withTrashed()->forceDelete();
                } else {
                    $branch->clientReceipts()->delete();
                }

                $branch->receiptCounters()->delete();

                // =========================================================
                // PaymentAllocation (عبر purchase_invoice_id)
                // =========================================================
                $branch->paymentAllocations()->delete();

                // =========================================================
                // Client accounting
                // =========================================================
                $branch->clientLedgerEntries()->delete();
                $branch->advanceInstallments()->delete();
                $branch->clientInstallmentPlans()->delete();
                $branch->clientAccounts()->delete();

                // =========================================================
                // Sales invoices
                // =========================================================
                if (in_array(\Illuminate\Database\Eloquent\SoftDeletes::class, class_uses_recursive(Invoice::class), true)) {
                    $branch->invoices()->withTrashed()->forceDelete();
                } else {
                    $branch->invoices()->delete();
                }

                // =========================================================
                // ✅ Purchasing + ✅ ProductPrice via PurchaseInvoiceItems
                // =========================================================
                $purchaseInvoiceIds = PurchaseInvoice::query()
                    ->withTrashed()
                    ->where('branch_id', $branch->id)
                    ->pluck('id');

                if ($purchaseInvoiceIds->isNotEmpty()) {

                    // 1) اجلب variants المستخدمة داخل عناصر فواتير شراء الفرع
                    $variantIds = PurchaseInvoiceItem::query()
                        ->whereIn('purchase_invoice_id', $purchaseInvoiceIds)
                        ->whereNotNull('product_variant_id')
                        ->pluck('product_variant_id')
                        ->unique()
                        ->values();

                    // 2) احذف product_prices المرتبطة بهذه الـ variants
                    // (لأن product_prices لا تملك branch_id)
                    if ($variantIds->isNotEmpty()) {
                        ProductPrice::query()
                            ->whereIn('product_variant_id', $variantIds)
                            ->delete();
                    }

                    // 3) حذف عناصر المرتجعات عبر purchase_return_id (بدون branch_id)
                    $purchaseReturnIds = PurchaseReturn::query()
                        ->withTrashed()
                        ->whereIn('purchase_invoice_id', $purchaseInvoiceIds)
                        ->pluck('id');

                    if ($purchaseReturnIds->isNotEmpty()) {
                        PurchaseReturnItem::query()
                            ->whereIn('purchase_return_id', $purchaseReturnIds)
                            ->delete();
                    }

                    // 4) حذف المرتجعات
                    PurchaseReturn::query()
                        ->whereIn('purchase_invoice_id', $purchaseInvoiceIds)
                        ->withTrashed()
                        ->forceDelete();

                    // 5) حذف تخصيصات الدفع عبر فواتير الشراء
                    PaymentAllocation::query()
                        ->whereIn('purchase_invoice_id', $purchaseInvoiceIds)
                        ->delete();

                    // 6) حذف عناصر فواتير الشراء
                    PurchaseInvoiceItem::query()
                        ->whereIn('purchase_invoice_id', $purchaseInvoiceIds)
                        ->delete();

                    // 7) حذف فواتير الشراء نفسها
                    PurchaseInvoice::query()
                        ->whereIn('id', $purchaseInvoiceIds)
                        ->withTrashed()
                        ->forceDelete();
                }

                // =========================================================
                // Stock (stock_balances عبر forBranch / stock_movements عبر warehouse->branches)
                // =========================================================
                StockBalance::query()->forBranch($branch->id)->delete();

                StockMovement::query()
                    ->whereHas('warehouse', function ($w) use ($branch) {
                        $w->whereHas('branches', fn ($b) => $b->where('branches.id', $branch->id));
                    })
                    ->delete();

                // =========================================================
                // Payroll (اعتماد على الفرع فقط عند توفر branch_id في الجداول)
                // =========================================================
                if (Schema::hasColumn('payroll_bonuses', 'branch_id')) {
                    $branch->payrollBonuses()->delete();
                }
                if (Schema::hasColumn('payroll_deductions', 'branch_id')) {
                    $branch->payrollDeductions()->delete();
                }

                $branch->payrolls()->delete();

                if (in_array(\Illuminate\Database\Eloquent\SoftDeletes::class, class_uses_recursive(PayrollRun::class), true)) {
                    $branch->payrollRuns()->withTrashed()->forceDelete();
                } else {
                    $branch->payrollRuns()->delete();
                }

                // =========================================================
                // HR
                // =========================================================
                $branch->employeeAdvances()->delete();
                $branch->workSchedules()->delete();
                $branch->overtimeRecords()->delete();
                $branch->payGrades()->delete();
                $branch->employees()->delete();

                // =========================================================
                // Attendance
                // =========================================================
                $branch->attendances()->delete();
                $branch->absences()->delete();

                // =========================================================
                // Expenses
                // =========================================================
                $branch->expenses()->delete();
                $branch->expenseCategories()->delete();

                // =========================================================
                // Suppliers
                // =========================================================
                $branch->supplierPayments()->delete();
                $branch->supplierLedgers()->delete();
                $branch->suppliers()->delete();

            });
        });
    }

    /* ================== الحالة ================== */

    public static function getStatuses(): array
    {
        return [
            self::STATUS_ACTIVE => 'نشط',
            self::STATUS_INACTIVE => 'غير نشط',
        ];
    }

    public function getStatusNameAttribute(): string
    {
        return self::getStatuses()[$this->status] ?? 'غير معروف';
    }

    /* ================== العلاقات الأساسية ================== */

    public function city()
    {
        return $this->belongsTo(City::class);
    }

    public function settings(): HasOne
    {
        return $this->hasOne(BranchSetting::class, 'branch_id');
    }

    public function clients()
    {
        return $this->belongsToMany(Client::class, 'branch_client')->withTimestamps();
    }

    public function warehouses()
    {
        return $this->belongsToMany(Warehouse::class, 'branch_warehouse', 'branch_id', 'warehouse_id')
            ->withPivot(['is_primary', 'assigned_at', 'unassigned_at'])
            ->withTimestamps();
    }

    public function primaryWarehouse()
    {
        return $this->belongsToMany(Warehouse::class, 'branch_warehouse', 'branch_id', 'warehouse_id')
            ->wherePivot('is_primary', true)
            ->wherePivotNull('unassigned_at')
            ->withTimestamps();
    }

    /* ================== HasMany ================== */

    public function employees(): HasMany
    {
        return $this->hasMany(Employee::class, 'branch_id');
    }

    public function purchaseInvoices(): HasMany
    {
        return $this->hasMany(PurchaseInvoice::class, 'branch_id');
    }

    public function invoices(): HasMany
    {
        return $this->hasMany(Invoice::class, 'branch_id');
    }

    public function attendances(): HasMany
    {
        return $this->hasMany(Attendance::class, 'branch_id');
    }

    public function absences(): HasMany
    {
        return $this->hasMany(Absence::class, 'branch_id');
    }

    public function clientAccounts(): HasMany
    {
        return $this->hasMany(ClientAccount::class, 'branch_id');
    }

    public function clientLedgerEntries(): HasMany
    {
        return $this->hasMany(ClientLedgerEntry::class, 'branch_id');
    }

    public function clientInstallmentPlans(): HasMany
    {
        return $this->hasMany(ClientInstallmentPlan::class, 'branch_id');
    }

    public function advanceInstallments(): HasMany
    {
        return $this->hasMany(AdvanceInstallment::class, 'branch_id');
    }

    public function clientReceipts(): HasMany
    {
        return $this->hasMany(ClientReceipt::class, 'branch_id');
    }

    public function receiptCounters(): HasMany
    {
        return $this->hasMany(ReceiptCounter::class, 'branch_id');
    }

    public function expenses(): HasMany
    {
        return $this->hasMany(Expense::class, 'branch_id');
    }

    public function expenseCategories(): HasMany
    {
        return $this->hasMany(ExpenseCategory::class, 'branch_id');
    }

    public function vaults(): HasMany
    {
        return $this->hasMany(Vault::class, 'branch_id');
    }

    public function suppliers(): HasMany
    {
        return $this->hasMany(Supplier::class, 'branch_id');
    }

    public function supplierLedgers(): HasMany
    {
        return $this->hasMany(SupplierLedger::class, 'branch_id');
    }

    public function supplierPayments(): HasMany
    {
        return $this->hasMany(SupplierPayment::class, 'branch_id');
    }

    public function workSchedules(): HasMany
    {
        return $this->hasMany(WorkSchedule::class, 'branch_id');
    }

    public function payGrades(): HasMany
    {
        return $this->hasMany(PayGrade::class, 'branch_id');
    }

    public function overtimeRecords(): HasMany
    {
        return $this->hasMany(OvertimeRecord::class, 'branch_id');
    }

    public function employeeAdvances(): HasMany
    {
        return $this->hasMany(EmployeeAdvance::class, 'branch_id');
    }

    public function payrollRuns(): HasMany
    {
        return $this->hasMany(PayrollRun::class, 'branch_id');
    }

    public function payrolls(): HasMany
    {
        return $this->hasMany(Payroll::class, 'branch_id');
    }

    public function payrollBonuses(): HasMany
    {
        return $this->hasMany(PayrollBonus::class, 'branch_id');
    }

    public function payrollDeductions(): HasMany
    {
        return $this->hasMany(PayrollDeduction::class, 'branch_id');
    }

    /* ================== HasManyThrough ================== */

    public function vaultTransactions(): HasManyThrough
    {
        return $this->hasManyThrough(
            VaultTransaction::class,
            Vault::class,
            'branch_id',
            'vault_id',
            'id',
            'id'
        );
    }

    public function receiptInvoiceAllocations(): HasManyThrough
    {
        return $this->hasManyThrough(
            ReceiptInvoiceAllocation::class,
            ClientReceipt::class,
            'branch_id',
            'client_receipt_id',
            'id',
            'id'
        );
    }

    public function paymentAllocations(): HasManyThrough
    {
        return $this->hasManyThrough(
            PaymentAllocation::class,
            PurchaseInvoice::class,
            'branch_id',
            'purchase_invoice_id',
            'id',
            'id'
        );
    }

    /* ================== Accessors للشعارات ================== */

    public function getLogoLightUrlAttribute(): ?string
    {
        return $this->settings?->logo_light_url;
    }

    public function getLogoDarkUrlAttribute(): ?string
    {
        return $this->settings?->logo_dark_url;
    }

    public function getFaviconUrlAttribute(): ?string
    {
        return $this->settings?->favicon_url;
    }

    /* ================== ألوان الفاتورة ================== */

    protected function defaultInvoiceCssVars(): array
    {
        return [
            '--primary' => '#C26D3C',
            '--primary-400' => '#D8925E',
            '--primary-700' => '#8A4A24',
            '--text' => '#0A3326',
            '--text-muted' => '#7E8B86',
            '--surface' => '#FFFFFF',
            '--surface-field' => '#E9ECEF',
            '--surface-pill' => '#F3F5F4',
            '--border' => '#DDE6E1',
        ];
    }

    public function getInvoiceCssVarsAttribute(): array
    {
        return $this->settings?->invoice_css_vars ?? $this->defaultInvoiceCssVars();
    }
}
