<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Validation\ValidationException;

class Barcode extends Model
{
    /*
     |----------------------------------------------------------------------
     | الثوابت الخاصة بحقل المصدر (source)
     |----------------------------------------------------------------------
     */
    public const SOURCE_INTERNAL     = 'internal';      // باركود داخلي من النظام / المستخدم
    public const SOURCE_VENDOR       = 'vendor';        // من المورد
    public const SOURCE_MANUFACTURER = 'manufacturer';  // من المصنع

    /**
     * القيم المسموح بها لحقل source
     */
    public const SOURCES = [
        self::SOURCE_INTERNAL,
        self::SOURCE_VENDOR,
        self::SOURCE_MANUFACTURER,
    ];

    protected $fillable = [
        'product_variant_id',
        'code',
        'is_primary',
        'source',
        'is_active',
        'user_id',
    ];

    protected $casts = [
        'is_primary' => 'boolean',
        'is_active'  => 'boolean',
    ];

    /**
     * القيمة الافتراضية لحقل المصدر
     */
    protected $attributes = [
        'source' => self::SOURCE_INTERNAL,
    ];

    protected static function booted(): void
    {
        static::saving(function (self $barcode) {
            /*
             * تنظيف كود الباركود من أي محارف غير رقمية
             */
            $barcode->code = preg_replace('/\D/', '', (string) $barcode->code);

            /*
             * التحقق من صحة كود الـ EAN-13
             */
            if (! self::validateEAN13($barcode->code)) {
                throw ValidationException::withMessages([
                    'code' => 'الباركود يجب أن يكون EAN-13 صحيحًا (13 رقمًا مع رقم تحقق صحيح).',
                ]);
            }

            /*
             * ضمان أن قيمة source ضمن القيم المسموح بها
             * وإن لم تكن، نرجعها إلى internal
             */
            if (! in_array($barcode->source, self::SOURCES, true)) {
                $barcode->source = self::SOURCE_INTERNAL;
            }

            /*
             * جعل هذا الباركود هو الأساسي الوحيد للنسخة
             */
            if ($barcode->is_primary) {
                static::where('product_variant_id', $barcode->product_variant_id)
                    ->whereKeyNot($barcode->id ?? 0)
                    ->update(['is_primary' => false]);
            }
        });

        static::deleted(function (self $barcode) {
            /*
             * لو حُذف الباركود الأساسي:
             * نختار أول باركود فعّال ونجعله أساسي
             */
            if ($barcode->is_primary) {
                $first = static::where('product_variant_id', $barcode->product_variant_id)
                    ->where('is_active', true)
                    ->first();

                if ($first) {
                    $first->update(['is_primary' => true]);
                }
            }
        });
    }

    /*
     |----------------------------------------------------------------------
     | دوال التحقق والتوليد EAN-13
     |----------------------------------------------------------------------
     */

    public static function validateEAN13(string $code): bool
    {
        if (strlen($code) !== 13 || ! ctype_digit($code)) {
            return false;
        }

        $checksum = 0;

        // أول 12 رقم لحساب رقم التحقق
        for ($i = 0; $i < 12; $i++) {
            $checksum += (int) $code[$i] * (($i % 2 === 0) ? 1 : 3);
        }

        $check = (10 - ($checksum % 10)) % 10;

        return $check === (int) $code[12];
    }

    public static function generateEAN13(): string
    {
        do {
            // مثال: بادئة محلية "62" + 10 أرقام عشوائية
            $base = '62' . str_pad((string) random_int(0, 9999999999), 10, '0', STR_PAD_LEFT);

            $sum = 0;
            for ($i = 0; $i < 12; $i++) {
                $sum += (int) $base[$i] * (($i % 2 === 0) ? 1 : 3);
            }

            $check = (10 - ($sum % 10)) % 10;
            $full  = $base . $check;
        } while (self::where('code', $full)->exists());

        return $full;
    }

    /*
     |----------------------------------------------------------------------
     | العلاقات
     |----------------------------------------------------------------------
     */

    public function variant(): BelongsTo
    {
        return $this->belongsTo(ProductVariant::class, 'product_variant_id');
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /*
     |----------------------------------------------------------------------
     | Scopes
     |----------------------------------------------------------------------
     */

    public function scopePrimary($q)
    {
        return $q->where('is_primary', true);
    }

    public function scopeActive($q)
    {
        return $q->where('is_active', true);
    }
}
