<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class AttributeValue extends Model
{
    protected $table = 'attribute_values';

    protected $fillable = [
        'attribute_id',
        'value',
        'display_value',
        'color_code',
        'sort_order',
        'is_active',
    ];

    protected $casts = [
        'is_active'  => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Hooks
     */
    protected static function booted(): void
    {
        static::saving(function (self $av) {
            // attribute_id مطلوب
            if (empty($av->attribute_id)) {
                throw new \InvalidArgumentException('attribute_id مطلوب لقيمة الخاصية.');
            }

            // توليد value من display_value عند غيابه
            if (empty($av->value) && ! empty($av->display_value)) {
                $av->value = str($av->display_value)->slug('_');
            }

            // تطبيع value (slug + حد أقصى للطول)
            if (! empty($av->value)) {
                $av->value = (string) str($av->value)
                    ->slug('_')
                    ->limit(100, '');
            }

            // sort_order تلقائي داخل نفس الخاصية
            if ($av->sort_order === null) {
                $av->sort_order = (int) static::where('attribute_id', $av->attribute_id)->max('sort_order') + 1;
            }

            // تطبيع كود اللون إن وُجد
            if (! empty($av->color_code)) {
                $c = strtoupper(trim($av->color_code));

                if (! str_starts_with($c, '#')) {
                    $c = "#{$c}";
                }

                if (! preg_match('/^#[0-9A-F]{6}$/', $c)) {
                    throw new \InvalidArgumentException('color_code يجب أن يكون بصيغة #RRGGBB.');
                }

                $av->color_code = $c;
            }
        });
    }

    /* ================= العلاقات ================= */

    /**
     * الخاصية المالكة لهذه القيمة (Color / Size ...)
     */
    public function attribute(): BelongsTo
    {
        return $this->belongsTo(Attribute::class, 'attribute_id');
    }

    /**
     * النسخ (ProductVariant) التي ترتبط بهذه القيمة عبر pivot: variant_attribute_values
     */
    public function variants(): BelongsToMany
    {
        return $this->belongsToMany(
            ProductVariant::class,
            'variant_attribute_values',
            'attribute_value_id', // FK في pivot يشير إلى attribute_values.id
            'variant_id'          // FK في pivot يشير إلى product_variants.id
        )->withPivot('attribute_id');
    }

    /* ================= السكوبات ================= */

    /**
     * قيَم مفعّلة فقط.
     */
    public function scopeActive($q)
    {
        return $q->where($this->getTable() . '.is_active', true);
    }

    /**
     * ترتيب افتراضي بحسب sort_order ثم id من جدول attribute_values تحديداً.
     */
    public function scopeOrdered($q)
    {
        $table = $this->getTable();

        return $q
            ->orderBy("{$table}.sort_order")
            ->orderBy("{$table}.id");
    }

    /**
     * تخصيص داخل خاصية محددة (نحدد الجدول لتفادي اللبس عند الـ join).
     */
    public function scopeForAttribute($q, int $attributeId)
    {
        $table = $this->getTable();

        return $q->where("{$table}.attribute_id", $attributeId);
    }

    /* ============== مساعدات عرض / خصائص محسوبة ============== */

    /**
     * وسم عرض مختصر "اسم الخاصية: القيمة"
     */
    public function getDisplayLabelAttribute(): string
    {
        $name = $this->attribute?->display_name ?: 'خاصية';
        return "{$name}: {$this->display_value}";
    }

    /**
     * هل هذه القيمة تابعة لخاصية نوعها "color"؟
     */
    public function getIsColorAttribute(): bool
    {
        return (string) ($this->attribute?->type) === 'color';
    }
}
