<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Carbon\Carbon;

class Attendance extends Model
{
    protected $fillable = [
        'branch_id',
        'employee_id',
        'work_schedule_id',
        'check_in',
        'check_out',
        'worked_minutes',
        'late_minutes',
        'overtime_minutes',
        'overtime_approved',
        'source',
        'status',
        'notes',
    ];

    protected $casts = [
        'check_in' => 'datetime',
        'check_out' => 'datetime',
        'overtime_approved' => 'boolean',
        'late_minutes' => 'integer',
        'overtime_minutes' => 'integer',
        'worked_minutes' => 'integer',
    ];

    // ==================== العلاقات ====================

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    public function workSchedule(): BelongsTo
    {
        return $this->belongsTo(WorkSchedule::class);
    }

    public function overtimeRecords(): HasMany
    {
        return $this->hasMany(OvertimeRecord::class);
    }

    // ==================== Accessors ====================

    /**
     * ساعات العمل الفعلية
     */
    public function getWorkedHoursAttribute(): ?float
    {
        if (!$this->worked_minutes) {
            return null;
        }
        return round($this->worked_minutes / 60, 2);
    }

    /**
     * ساعات التأخير
     */
    public function getLateHoursAttribute(): float
    {
        return round($this->late_minutes / 60, 2);
    }

    /**
     * ساعات الوقت الإضافي
     */
    public function getOvertimeHoursAttribute(): float
    {
        return round($this->overtime_minutes / 60, 2);
    }

    /**
     * تاريخ الحضور فقط
     */
    public function getCheckInDateAttribute(): ?string
    {
        return $this->check_in?->format('Y-m-d');
    }

    /**
     * وقت الحضور فقط
     */
    public function getCheckInTimeAttribute(): ?string
    {
        return $this->check_in?->format('H:i');
    }

    /**
     * وقت الانصراف فقط
     */
    public function getCheckOutTimeAttribute(): ?string
    {
        return $this->check_out?->format('H:i');
    }

    /**
     * وقت الحضور منسق (12 ساعة)
     */
    public function getCheckInTime12hAttribute(): ?string
    {
        return $this->check_in?->format('h:i A');
    }

    /**
     * وقت الانصراف منسق (12 ساعة)
     */
    public function getCheckOutTime12hAttribute(): ?string
    {
        return $this->check_out?->format('h:i A');
    }

    /**
     * هل الحضور مكتمل؟
     */
    public function getIsCompleteAttribute(): bool
    {
        return !is_null($this->check_in) && !is_null($this->check_out);
    }

    /**
     * هل الحضور غير مكتمل؟
     */
    public function getIsIncompleteAttribute(): bool
    {
        return !is_null($this->check_in) && is_null($this->check_out);
    }

    /**
     * هل متأخر؟
     */
    public function getIsLateAttribute(): bool
    {
        return $this->late_minutes > 0;
    }

    /**
     * هل لديه وقت إضافي؟
     */
    public function getHasOvertimeAttribute(): bool
    {
        return $this->overtime_minutes > 0;
    }

    /**
     * نص الحالة بالعربي
     */
    public function getStatusLabelAttribute(): string
    {
        return match ($this->status) {
            'pending' => 'قيد المراجعة',
            'confirmed' => 'مؤكد',
            'rejected' => 'مرفوض',
            default => 'غير معروف',
        };
    }

    /**
     * لون الحالة
     */
    public function getStatusColorAttribute(): string
    {
        return match ($this->status) {
            'pending' => 'warning',
            'confirmed' => 'success',
            'rejected' => 'danger',
            default => 'gray',
        };
    }

    /**
     * نص المصدر بالعربي
     */
    public function getSourceLabelAttribute(): string
    {
        return match ($this->source) {
            'manual' => 'يدوي',
            'device' => 'جهاز بصمة',
            'import' => 'استيراد',
            default => 'غير معروف',
        };
    }

    /**
     * أيقونة المصدر
     */
    public function getSourceIconAttribute(): string
    {
        return match ($this->source) {
            'manual' => 'heroicon-o-pencil',
            'device' => 'heroicon-o-finger-print',
            'import' => 'heroicon-o-arrow-down-tray',
            default => 'heroicon-o-question-mark-circle',
        };
    }

    // ==================== دوال الحساب ====================

    /**
     * حساب جميع القيم تلقائياً
     */
    public function calculateAll(): void
    {
        if (!$this->check_in || !$this->workSchedule) {
            return;
        }

        // حساب التأخير
        $this->late_minutes = $this->workSchedule->calculateLateMinutes($this->check_in);

        // حساب دقائق ووقت العمل إذا كان هناك انصراف
        if ($this->check_out) {
            $this->worked_minutes = $this->workSchedule->calculateWorkedMinutes(
                $this->check_in,
                $this->check_out
            );

            // حساب الوقت الإضافي
            $this->overtime_minutes = $this->workSchedule->calculateOvertimeMinutes($this->check_out);
        }

        $this->save();
    }

    /**
     * حساب تلقائي عند الانصراف
     */
    public function autoCalculateOnCheckOut(): void
    {
        if ($this->check_out && $this->workSchedule) {
            $this->worked_minutes = $this->workSchedule->calculateWorkedMinutes(
                $this->check_in,
                $this->check_out
            );
            $this->overtime_minutes = $this->workSchedule->calculateOvertimeMinutes($this->check_out);
            $this->save();
        }
    }

    // ==================== دوال العمليات ====================

    /**
     * تسجيل الحضور
     */
    public function checkIn(?Carbon $time = null): void
    {
        $checkInTime = $time ?? now();

        $this->check_in = $checkInTime;

        // حساب التأخير
        if ($this->workSchedule) {
            $this->late_minutes = $this->workSchedule->calculateLateMinutes($checkInTime);
        }

        $this->save();
    }

    /**
     * تسجيل الانصراف
     */
    public function checkOut(?Carbon $time = null): void
    {
        $checkOutTime = $time ?? now();

        $this->check_out = $checkOutTime;
        $this->autoCalculateOnCheckOut();
    }

    /**
     * تأكيد الحضور
     */
    public function confirm(): void
    {
        $this->status = 'confirmed';
        $this->save();
    }

    /**
     * رفض الحضور
     */
    public function reject(): void
    {
        $this->status = 'rejected';
        $this->save();
    }

    /**
     * جعله قيد المراجعة
     */
    public function markAsPending(): void
    {
        $this->status = 'pending';
        $this->save();
    }

    /**
     * الموافقة على الوقت الإضافي
     */
    public function approveOvertime(): void
    {
        $this->overtime_approved = true;
        $this->save();
    }

    /**
     * إلغاء موافقة الوقت الإضافي
     */
    public function unapproveOvertime(): void
    {
        $this->overtime_approved = false;
        $this->save();
    }

    // ==================== Scopes ====================

    /**
     * لموظف معين
     */
    public function scopeForEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    /**
     * لفرع معين
     */
    public function scopeForBranch($query, $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    /**
     * لتاريخ معين
     */
    public function scopeForDate($query, $date)
    {
        return $query->whereDate('check_in', $date);
    }

    /**
     * لشهر معين
     */
    public function scopeForMonth($query, int $year, int $month)
    {
        return $query->whereYear('check_in', $year)
            ->whereMonth('check_in', $month);
    }

    /**
     * بين تاريخين
     */
    public function scopeBetweenDates($query, $startDate, $endDate)
    {
        return $query->whereBetween('check_in', [$startDate, $endDate]);
    }

    /**
     * الحضور المكتمل (حضور وانصراف)
     */
    public function scopeComplete($query)
    {
        return $query->whereNotNull('check_in')
            ->whereNotNull('check_out');
    }

    /**
     * الحضور غير المكتمل (حضور فقط بدون انصراف)
     */
    public function scopeIncomplete($query)
    {
        return $query->whereNotNull('check_in')
            ->whereNull('check_out');
    }

    /**
     * المتأخرين
     */
    public function scopeLate($query)
    {
        return $query->where('late_minutes', '>', 0);
    }

    /**
     * غير المتأخرين
     */
    public function scopeOnTime($query)
    {
        return $query->where('late_minutes', 0);
    }

    /**
     * لديهم وقت إضافي
     */
    public function scopeWithOvertime($query)
    {
        return $query->where('overtime_minutes', '>', 0);
    }

    /**
     * الوقت الإضافي المعتمد
     */
    public function scopeOvertimeApproved($query)
    {
        return $query->where('overtime_approved', true)
            ->where('overtime_minutes', '>', 0);
    }

    /**
     * الوقت الإضافي قيد الانتظار
     */
    public function scopeOvertimePending($query)
    {
        return $query->where('overtime_approved', false)
            ->where('overtime_minutes', '>', 0);
    }

    /**
     * الحضور المؤكد
     */
    public function scopeConfirmed($query)
    {
        return $query->where('status', 'confirmed');
    }

    /**
     * الحضور قيد المراجعة
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * الحضور المرفوض
     */
    public function scopeRejected($query)
    {
        return $query->where('status', 'rejected');
    }

    /**
     * الحضور اليدوي
     */
    public function scopeManual($query)
    {
        return $query->where('source', 'manual');
    }

    /**
     * الحضور من الجهاز
     */
    public function scopeFromDevice($query)
    {
        return $query->where('source', 'device');
    }

    /**
     * الحضور المستورد
     */
    public function scopeImported($query)
    {
        return $query->where('source', 'import');
    }

    /**
     * اليوم
     */
    public function scopeToday($query)
    {
        return $query->whereDate('check_in', today());
    }

    /**
     * أمس
     */
    public function scopeYesterday($query)
    {
        return $query->whereDate('check_in', today()->subDay());
    }

    /**
     * هذا الأسبوع
     */
    public function scopeThisWeek($query)
    {
        return $query->whereBetween('check_in', [
            now()->startOfWeek(),
            now()->endOfWeek()
        ]);
    }

    /**
     * هذا الشهر
     */
    public function scopeThisMonth($query)
    {
        return $query->whereYear('check_in', now()->year)
            ->whereMonth('check_in', now()->month);
    }
}
