<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Absence extends Model
{
    protected $fillable = [
        'branch_id',
        'employee_id',
        'date',
        'type',
        'minutes_expected',
        'minutes_excused',
        'notes',
    ];

    protected $casts = [
        'date' => 'date',
        'minutes_expected' => 'integer',
        'minutes_excused' => 'integer',
    ];

    // ==================== العلاقات ====================

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    // ==================== Accessors ====================

    /**
     * الدقائق المخصومة (المتوقعة - المعذور عنها)
     */
    public function getMinutesDeductedAttribute(): int
    {
        return max(0, $this->minutes_expected - $this->minutes_excused);
    }

    /**
     * الساعات المتوقعة
     */
    public function getHoursExpectedAttribute(): float
    {
        return round($this->minutes_expected / 60, 2);
    }

    /**
     * الساعات المعذور عنها
     */
    public function getHoursExcusedAttribute(): float
    {
        return round($this->minutes_excused / 60, 2);
    }

    /**
     * الساعات المخصومة
     */
    public function getHoursDeductedAttribute(): float
    {
        return round($this->minutes_deducted / 60, 2);
    }

    /**
     * نسبة الإعذار
     */
    public function getExcusePercentageAttribute(): float
    {
        if ($this->minutes_expected == 0) {
            return 0;
        }
        return round(($this->minutes_excused / $this->minutes_expected) * 100, 2);
    }

    /**
     * هل معذور بالكامل؟
     */
    public function getIsFullyExcusedAttribute(): bool
    {
        return $this->minutes_excused >= $this->minutes_expected;
    }

    /**
     * هل معذور جزئياً؟
     */
    public function getIsPartiallyExcusedAttribute(): bool
    {
        return $this->minutes_excused > 0 && $this->minutes_excused < $this->minutes_expected;
    }

    /**
     * هل غير معذور؟
     */
    public function getIsUnexcusedAttribute(): bool
    {
        return $this->minutes_excused == 0;
    }

    /**
     * نص نوع الغياب بالعربي
     */
    public function getTypeLabelAttribute(): string
    {
        return match ($this->type) {
            'unpaid' => 'غياب غير مدفوع',
            'paid' => 'إجازة مدفوعة',
            'sick' => 'إجازة مرضية',
            'vacation' => 'إجازة سنوية',
            'other' => 'أخرى',
            default => 'غير معروف',
        };
    }

    /**
     * لون نوع الغياب
     */
    public function getTypeColorAttribute(): string
    {
        return match ($this->type) {
            'unpaid' => 'danger',
            'paid' => 'success',
            'sick' => 'warning',
            'vacation' => 'info',
            'other' => 'gray',
            default => 'gray',
        };
    }

    /**
     * أيقونة نوع الغياب
     */
    public function getTypeIconAttribute(): string
    {
        return match ($this->type) {
            'unpaid' => 'heroicon-o-x-circle',
            'paid' => 'heroicon-o-check-circle',
            'sick' => 'heroicon-o-heart',
            'vacation' => 'heroicon-o-sun',
            'other' => 'heroicon-o-question-mark-circle',
            default => 'heroicon-o-minus-circle',
        };
    }

    /**
     * التاريخ منسق
     */
    public function getFormattedDateAttribute(): string
    {
        return $this->date->format('Y-m-d');
    }

    /**
     * اسم اليوم بالعربي
     */
    public function getDayNameAttribute(): string
    {
        $days = ['الأحد', 'الإثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت'];
        return $days[$this->date->dayOfWeek] ?? '';
    }

    // ==================== دوال الحساب ====================

    /**
     * حساب مبلغ الخصم بناءً على الراتب اليومي
     */
    public function calculateDeduction(int $dailySalary): int
    {
        if ($this->is_fully_excused || $this->type === 'paid') {
            return 0;
        }

        $deductedHours = $this->hours_deducted;
        $hourlyRate = $dailySalary / 8; // افتراض 8 ساعات عمل يومياً

        return (int) ($deductedHours * $hourlyRate);
    }

    /**
     * حساب مبلغ الخصم بدقة أكبر (حسب ساعات العمل الفعلية)
     */
    public function calculateDeductionPrecise(int $dailySalary, int $dailyWorkHours = 8): int
    {
        if ($this->is_fully_excused || $this->type === 'paid') {
            return 0;
        }

        $deductedHours = $this->hours_deducted;
        $hourlyRate = $dailySalary / $dailyWorkHours;

        return (int) ($deductedHours * $hourlyRate);
    }

    // ==================== دوال العمليات ====================

    /**
     * الإعذار بالكامل
     */
    public function fullyExcuse(): void
    {
        $this->update([
            'minutes_excused' => $this->minutes_expected,
        ]);
    }

    /**
     * إلغاء الإعذار
     */
    public function removeExcuse(): void
    {
        $this->update([
            'minutes_excused' => 0,
        ]);
    }

    /**
     * إعذار جزئي بعدد ساعات
     */
    public function excuseHours(float $hours): void
    {
        $minutes = (int) ($hours * 60);
        $this->update([
            'minutes_excused' => min($minutes, $this->minutes_expected),
        ]);
    }

    /**
     * إعذار جزئي بعدد دقائق
     */
    public function excuseMinutes(int $minutes): void
    {
        $this->update([
            'minutes_excused' => min($minutes, $this->minutes_expected),
        ]);
    }

    // ==================== Scopes ====================

    /**
     * لموظف معين
     */
    public function scopeForEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    /**
     * لفرع معين
     */
    public function scopeForBranch($query, $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    /**
     * لتاريخ معين
     */
    public function scopeForDate($query, $date)
    {
        return $query->whereDate('date', $date);
    }

    /**
     * لشهر معين
     */
    public function scopeForMonth($query, int $year, int $month)
    {
        return $query->whereYear('date', $year)
            ->whereMonth('date', $month);
    }

    /**
     * بين تاريخين
     */
    public function scopeBetweenDates($query, $startDate, $endDate)
    {
        return $query->whereBetween('date', [$startDate, $endDate]);
    }

    /**
     * غياب غير مدفوع
     */
    public function scopeUnpaid($query)
    {
        return $query->where('type', 'unpaid');
    }

    /**
     * إجازة مدفوعة
     */
    public function scopePaid($query)
    {
        return $query->where('type', 'paid');
    }

    /**
     * إجازة مرضية
     */
    public function scopeSick($query)
    {
        return $query->where('type', 'sick');
    }

    /**
     * إجازة سنوية
     */
    public function scopeVacation($query)
    {
        return $query->where('type', 'vacation');
    }

    /**
     * أخرى
     */
    public function scopeOther($query)
    {
        return $query->where('type', 'other');
    }

    /**
     * معذور بالكامل
     */
    public function scopeFullyExcused($query)
    {
        return $query->whereRaw('minutes_excused >= minutes_expected');
    }

    /**
     * معذور جزئياً
     */
    public function scopePartiallyExcused($query)
    {
        return $query->where('minutes_excused', '>', 0)
            ->whereRaw('minutes_excused < minutes_expected');
    }

    /**
     * غير معذور
     */
    public function scopeUnexcused($query)
    {
        return $query->where('minutes_excused', 0);
    }

    /**
     * سيتم خصم منه (غير معذور بالكامل وليس مدفوع)
     */
    public function scopeWillBeDeducted($query)
    {
        return $query->whereRaw('minutes_excused < minutes_expected')
            ->where('type', '!=', 'paid');
    }

    /**
     * لن يتم خصم منه (معذور بالكامل أو مدفوع)
     */
    public function scopeWontBeDeducted($query)
    {
        return $query->where(function ($q) {
            $q->whereRaw('minutes_excused >= minutes_expected')
                ->orWhere('type', 'paid');
        });
    }

    /**
     * اليوم
     */
    public function scopeToday($query)
    {
        return $query->whereDate('date', today());
    }

    /**
     * هذا الأسبوع
     */
    public function scopeThisWeek($query)
    {
        return $query->whereBetween('date', [
            now()->startOfWeek(),
            now()->endOfWeek()
        ]);
    }

    /**
     * هذا الشهر
     */
    public function scopeThisMonth($query)
    {
        return $query->whereYear('date', now()->year)
            ->whereMonth('date', now()->month);
    }
}
