<?php

namespace App\Livewire\Store;

use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\Wishlist;
use Illuminate\Support\Collection;
use Livewire\Attributes\On;
use Livewire\Component;

class WishlistManager extends Component
{
    public function getItemsProperty(): Collection
    {
        $client = auth('client')->user();

        if (! $client) {
            return collect([]);
        }

        return Wishlist::query()
            ->with(['product.variants' => fn ($q) => $q->where('is_active', true)])
            ->where('client_id', $client->id)
            ->get()
            ->map(function (Wishlist $wishlist) {
                $product = $wishlist->product;
                $variant = $wishlist->variant ?? $product->variants->firstWhere('is_default', true) ?? $product->variants->first();
                $price = $this->calculatePrice($product, $variant);

                return [
                    'id' => $wishlist->id,
                    'product_id' => $product->id,
                    'variant_id' => $variant?->id,
                    'name' => $product->name,
                    'image' => $product->main_image_url ?? $product->image_url,
                    'price' => $price,
                    'slug' => $product->slug,
                    'in_stock' => $variant ? $variant->total_available > 0 : false,
                ];
            });
    }

    public function getCountProperty(): int
    {
        $client = auth('client')->user();

        if (! $client) {
            return 0;
        }

        return Wishlist::where('client_id', $client->id)->count();
    }

    public function getProductIdsProperty(): array
    {
        $client = auth('client')->user();

        if (! $client) {
            return [];
        }

        return Wishlist::where('client_id', $client->id)
            ->pluck('product_id')
            ->toArray();
    }

    #[On('wishlist:toggle')]
    public function toggleWishlist(int $productId, ?int $variantId = null): void
    {
        $client = auth('client')->user();

        if (! $client) {
            return;
        }

        $existing = Wishlist::query()
            ->where('client_id', $client->id)
            ->where('product_id', $productId)
            ->first();

        if ($existing) {
            $existing->delete();
        } else {
            Wishlist::create([
                'client_id' => $client->id,
                'product_id' => $productId,
                'product_variant_id' => $variantId,
            ]);
        }

        $this->dispatch('wishlist:updated');
    }

    public function removeItem(int $wishlistId): void
    {
        $client = auth('client')->user();

        if (! $client) {
            return;
        }

        Wishlist::where('id', $wishlistId)
            ->where('client_id', $client->id)
            ->delete();

        $this->dispatch('wishlist:updated');
    }

    public function clearWishlist(): void
    {
        $client = auth('client')->user();

        if (! $client) {
            return;
        }

        Wishlist::where('client_id', $client->id)->delete();

        $this->dispatch('wishlist:updated');
    }

    public function addToCart(int $wishlistId): void
    {
        $client = auth('client')->user();

        if (! $client) {
            return;
        }

        $wishlist = Wishlist::where('id', $wishlistId)
            ->where('client_id', $client->id)
            ->with('product.variants')
            ->first();

        if (! $wishlist) {
            return;
        }

        $product = $wishlist->product;
        $variant = $wishlist->variant ?? $product->variants->firstWhere('is_default', true) ?? $product->variants->first();

        if ($variant) {
            $this->dispatch('cart:add', id: $product->id, qty: 1, variantId: $variant->id);
        }
    }

    public function moveAllToCart(): void
    {
        $items = $this->items;

        foreach ($items as $item) {
            if ($item['variant_id']) {
                $this->dispatch('cart:add', id: $item['product_id'], qty: 1, variantId: $item['variant_id']);
            }
        }

        $this->clearWishlist();
    }

    public function isInWishlist(int $productId): bool
    {
        return in_array($productId, $this->productIds);
    }

    protected function calculatePrice(Product $product, $variant): int
    {
        if (! $variant) {
            return 0;
        }

        $client = auth('client')->user();
        $priceType = 'retail';

        if ($client) {
            $t = mb_strtolower((string) ($client->client_type ?? ''));
            $hasCompanyName = trim((string) ($client->company ?? '')) !== '';
            $isCompany = $hasCompanyName || str_contains($t, 'company') || str_contains($t, 'شركة');
            $priceType = $isCompany ? 'wholesale' : 'retail';
        }

        $discountPercent = (int) ($product->discount_percent ?? 0);

        $row = ProductPrice::query()
            ->where('product_id', $product->id)
            ->where('product_variant_id', $variant->id)
            ->first();

        $current = 0;

        if ($row) {
            $current = $priceType === 'wholesale'
                ? (int) ($row->wholesale_price ?? 0)
                : (int) ($row->retail_price ?? 0);
        }

        if ($current <= 0) {
            $cost = (int) ($variant->cost ?? 0);
            $margin = $priceType === 'wholesale' ? 10.0 : 20.0;
            $current = (int) round($cost * (1 + $margin / 100));
        }

        if ($discountPercent > 0) {
            $current = (int) round(($current * (100 - $discountPercent)) / 100);
        }

        return $current;
    }

    public function formatPrice(int $value): string
    {
        return number_format($value, 0, '.', ',').' د.ع';
    }

    public function render()
    {
        return view('livewire.store.wishlist-manager', [
            'items' => $this->items,
            'count' => $this->count,
            'productIds' => $this->productIds,
        ]);
    }
}
