<?php

namespace App\Livewire\Store;

use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

class ProductGrid extends Component
{
    use WithPagination;

    #[Url(as: 'category')]
    public ?int $categoryId = null;

    #[Url(as: 'brand')]
    public ?int $brandId = null;

    #[Url(as: 'q')]
    public string $search = '';

    #[Url]
    public string $sort = 'newest';

    #[Url(as: 'view')]
    public string $viewMode = 'grid';

    #[Url(as: 'cols')]
    public int $gridColumns = 3;

    public int $perPage = 24;

    public bool $saleOnly = false;

    public bool $filtersOpen = false;

    /**
     * Initial category ID passed from parent (not synced to URL).
     */
    public ?int $initialCategoryId = null;

    protected $queryString = [
        'categoryId' => ['except' => null, 'as' => 'category'],
        'brandId' => ['except' => null, 'as' => 'brand'],
        'search' => ['except' => '', 'as' => 'q'],
        'sort' => ['except' => 'newest'],
        'viewMode' => ['except' => 'grid', 'as' => 'view'],
        'gridColumns' => ['except' => 3, 'as' => 'cols'],
    ];

    public function mount(?int $categoryId = null): void
    {
        // If a category ID is passed as a parameter, use it as the initial filter
        if ($categoryId !== null && $this->categoryId === null) {
            $this->categoryId = $categoryId;
            $this->initialCategoryId = $categoryId;
        }
    }

    public function updatingSearch(): void
    {
        $this->resetPage();
    }

    public function updatingCategoryId(): void
    {
        $this->resetPage();
    }

    public function updatingBrandId(): void
    {
        $this->resetPage();
    }

    public function updatingSort(): void
    {
        $this->resetPage();
    }

    public function getCategoriesProperty(): Collection
    {
        return Category::query()
            ->withCount(['products' => fn ($q) => $q->active()])
            ->orderBy('name')
            ->get();
    }

    public function getBrandsProperty(): Collection
    {
        return Brand::query()
            ->withCount(['products' => fn ($q) => $q->active()])
            ->orderBy('name')
            ->get();
    }

    public function getProductsProperty(): LengthAwarePaginator
    {
        $query = Product::query()
            ->active()
            ->with(['category', 'brand', 'defaultVariant', 'variants' => fn ($q) => $q->where('is_active', true)]);

        // Filter by category
        if ($this->categoryId) {
            $query->where('category_id', $this->categoryId);
        }

        // Filter by brand
        if ($this->brandId) {
            $query->where('brand_id', $this->brandId);
        }

        // Filter by search term
        if (trim($this->search) !== '') {
            $search = $this->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Note: Sale filtering removed - discount_percent doesn't exist on products table

        // Sorting
        switch ($this->sort) {
            case 'price_asc':
                $query->orderByRaw('(SELECT MIN(cost) FROM product_variants WHERE product_id = products.id AND is_active = 1) ASC');
                break;
            case 'price_desc':
                $query->orderByRaw('(SELECT MIN(cost) FROM product_variants WHERE product_id = products.id AND is_active = 1) DESC');
                break;
            case 'name_asc':
                $query->orderBy('name', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;
            case 'newest':
            default:
                $query->latest('id');
                break;
        }

        return $query->paginate($this->perPage);
    }

    public function getSelectedCategoryProperty(): ?Category
    {
        return $this->categoryId ? Category::find($this->categoryId) : null;
    }

    public function getSelectedBrandProperty(): ?Brand
    {
        return $this->brandId ? Brand::find($this->brandId) : null;
    }

    public function selectCategory(?int $id): void
    {
        $this->categoryId = $id;
        $this->resetPage();
    }

    public function selectBrand(?int $id): void
    {
        $this->brandId = $id;
        $this->resetPage();
    }

    public function setSort(string $sort): void
    {
        $this->sort = $sort;
        $this->resetPage();
    }

    public function setViewMode(string $mode): void
    {
        $this->viewMode = $mode;
    }

    public function setGridColumns(int $cols): void
    {
        $this->gridColumns = max(2, min(5, $cols));
        $this->viewMode = 'grid';
    }

    public function toggleSaleOnly(): void
    {
        $this->saleOnly = ! $this->saleOnly;
        $this->resetPage();
    }

    public function toggleFilters(): void
    {
        $this->filtersOpen = ! $this->filtersOpen;
    }

    public function clearFilters(): void
    {
        $this->categoryId = null;
        $this->brandId = null;
        $this->search = '';
        $this->saleOnly = false;
        $this->resetPage();
    }

    public function hasActiveFilters(): bool
    {
        return $this->categoryId !== null
            || $this->brandId !== null
            || trim($this->search) !== ''
            || $this->saleOnly;
    }

    public function render()
    {
        return view('livewire.store.product-grid', [
            'products' => $this->products,
            'categories' => $this->categories,
            'brands' => $this->brands,
            'selectedCategory' => $this->selectedCategory,
            'selectedBrand' => $this->selectedBrand,
            'hasFilters' => $this->hasActiveFilters(),
        ]);
    }
}
