<?php

namespace App\Livewire\Store;

use App\Models\Compare;
use App\Models\Product;
use App\Models\ProductPrice;
use Illuminate\Support\Collection;
use Livewire\Attributes\On;
use Livewire\Component;

class CompareManager extends Component
{
    public int $maxProducts = 4;

    protected function getSessionIdentifier(): array
    {
        $client = auth('client')->user();
        $sessionId = session()->getId();

        return $client
            ? ['client_id' => $client->id]
            : ['session_id' => $sessionId];
    }

    public function getItemsProperty(): Collection
    {
        $identifier = $this->getSessionIdentifier();

        return Compare::query()
            ->with(['product.variants' => fn ($q) => $q->where('is_active', true), 'product.category'])
            ->where($identifier)
            ->get()
            ->map(function (Compare $compare) {
                $product = $compare->product;
                $variant = $compare->variant ?? $product->variants->firstWhere('is_default', true) ?? $product->variants->first();
                $price = $this->calculatePrice($product, $variant);

                return [
                    'id' => $compare->id,
                    'product_id' => $product->id,
                    'variant_id' => $variant?->id,
                    'name' => $product->name,
                    'image' => $product->main_image_url ?? $product->image_url,
                    'category' => $product->category?->name ?? '—',
                    'price' => $price,
                    'slug' => $product->slug,
                    'in_stock' => $variant ? $variant->total_available > 0 : false,
                    'rating' => $product->avg_rating ?? null,
                    'description' => $product->short_description ?? $product->description,
                ];
            });
    }

    public function getCountProperty(): int
    {
        $identifier = $this->getSessionIdentifier();

        return Compare::where($identifier)->count();
    }

    public function getProductIdsProperty(): array
    {
        $identifier = $this->getSessionIdentifier();

        return Compare::where($identifier)
            ->pluck('product_id')
            ->toArray();
    }

    #[On('compare:toggle')]
    public function toggleCompare(int $productId, ?int $variantId = null): void
    {
        $identifier = $this->getSessionIdentifier();

        $existing = Compare::query()
            ->where($identifier)
            ->where('product_id', $productId)
            ->first();

        if ($existing) {
            $existing->delete();
        } else {
            // Check max limit
            $currentCount = Compare::where($identifier)->count();

            if ($currentCount >= $this->maxProducts) {
                return;
            }

            Compare::create(array_merge($identifier, [
                'product_id' => $productId,
                'product_variant_id' => $variantId,
            ]));
        }

        $this->dispatch('compare:updated');
    }

    public function removeItem(int $compareId): void
    {
        $identifier = $this->getSessionIdentifier();

        Compare::where('id', $compareId)
            ->where($identifier)
            ->delete();

        $this->dispatch('compare:updated');
    }

    public function clearCompare(): void
    {
        $identifier = $this->getSessionIdentifier();

        Compare::where($identifier)->delete();

        $this->dispatch('compare:updated');
    }

    public function addToCart(int $compareId): void
    {
        $identifier = $this->getSessionIdentifier();

        $compare = Compare::where('id', $compareId)
            ->where($identifier)
            ->with('product.variants')
            ->first();

        if (! $compare) {
            return;
        }

        $product = $compare->product;
        $variant = $compare->variant ?? $product->variants->firstWhere('is_default', true) ?? $product->variants->first();

        if ($variant) {
            $this->dispatch('cart:add', id: $product->id, qty: 1, variantId: $variant->id);
        }
    }

    public function isInCompare(int $productId): bool
    {
        return in_array($productId, $this->productIds);
    }

    protected function calculatePrice(Product $product, $variant): int
    {
        if (! $variant) {
            return 0;
        }

        $client = auth('client')->user();
        $priceType = 'retail';

        if ($client) {
            $t = mb_strtolower((string) ($client->client_type ?? ''));
            $hasCompanyName = trim((string) ($client->company ?? '')) !== '';
            $isCompany = $hasCompanyName || str_contains($t, 'company') || str_contains($t, 'شركة');
            $priceType = $isCompany ? 'wholesale' : 'retail';
        }

        $discountPercent = (int) ($product->discount_percent ?? 0);

        $row = ProductPrice::query()
            ->where('product_id', $product->id)
            ->where('product_variant_id', $variant->id)
            ->first();

        $current = 0;

        if ($row) {
            $current = $priceType === 'wholesale'
                ? (int) ($row->wholesale_price ?? 0)
                : (int) ($row->retail_price ?? 0);
        }

        if ($current <= 0) {
            $cost = (int) ($variant->cost ?? 0);
            $margin = $priceType === 'wholesale' ? 10.0 : 20.0;
            $current = (int) round($cost * (1 + $margin / 100));
        }

        if ($discountPercent > 0) {
            $current = (int) round(($current * (100 - $discountPercent)) / 100);
        }

        return $current;
    }

    public function formatPrice(int $value): string
    {
        return number_format($value, 0, '.', ',').' د.ع';
    }

    public function render()
    {
        return view('livewire.store.compare-manager', [
            'items' => $this->items,
            'count' => $this->count,
            'productIds' => $this->productIds,
            'maxProducts' => $this->maxProducts,
        ]);
    }
}
