<?php

namespace App\Livewire\Store;

use App\Models\Cart;
use App\Models\CartItem;
use App\Models\Coupon;
use App\Models\Currency;
use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\ProductVariant;
use Illuminate\Support\Collection;
use Livewire\Attributes\On;
use Livewire\Component;

class CartManager extends Component
{
    public int $freeShippingThreshold = 100000;

    public int $shippingFee = 5000;

    protected function getCart(): Cart
    {
        $client = auth('client')->user();
        $sessionId = session()->getId();

        return Cart::query()
            ->with(['items.product', 'items.variant'])
            ->active()
            ->when($client, fn ($q) => $q->where('client_id', $client->id))
            ->when(! $client, fn ($q) => $q->where('session_id', $sessionId))
            ->firstOrCreate(
                $client
                    ? ['client_id' => $client->id]
                    : ['session_id' => $sessionId],
                ['currency_id' => Currency::getDefaultId()]
            );
    }

    public function getItemsProperty(): Collection
    {
        return $this->getCart()->items->map(function (CartItem $item) {
            $product = $item->product;
            $variant = $item->variant;

            return [
                'id' => $item->id,
                'product_id' => $product->id,
                'variant_id' => $variant?->id,
                'name' => $product->name,
                'image' => $product->thumbnail_thumb_url ?? $product->thumbnail_url,
                'variant_name' => $variant?->name ?? $variant?->sku,
                'quantity' => $item->quantity,
                'unit_price' => $item->unit_price,
                'line_total' => $item->line_total,
                'slug' => $product->slug,
            ];
        });
    }

    public function getCountProperty(): int
    {
        return $this->getCart()->items_count;
    }

    public function getSubtotalProperty(): int
    {
        return $this->getCart()->subtotal;
    }

    public function getCalculatedShippingProperty(): int
    {
        return $this->subtotal >= $this->freeShippingThreshold ? 0 : $this->shippingFee;
    }

    public function getTotalProperty(): int
    {
        return $this->subtotal + $this->calculatedShipping;
    }

    public function getShippingProgressProperty(): int
    {
        if ($this->subtotal >= $this->freeShippingThreshold) {
            return 100;
        }

        return (int) round(($this->subtotal / $this->freeShippingThreshold) * 100);
    }

    public function getRemainingForFreeShippingProperty(): int
    {
        return max(0, $this->freeShippingThreshold - $this->subtotal);
    }

    #[On('cart:update-quantity')]
    public function handleUpdateQuantity(int $itemId, int $quantity): void
    {
        $this->updateQuantity($itemId, $quantity);
    }

    #[On('cart:remove-item')]
    public function handleRemoveItem(int $itemId): void
    {
        $this->removeItem($itemId);
    }

    #[On('cart:clear')]
    public function handleClearCart(): void
    {
        $this->clearCart();
    }

    #[On('cart:add')]
    public function addToCart(int $id, int $qty = 1, ?int $variantId = null): void
    {
        $product = Product::with(['variants' => fn ($q) => $q->where('is_active', true)])->findOrFail($id);

        // Get variant
        if ($variantId) {
            $variant = $product->variants->firstWhere('id', $variantId);
        } else {
            $variant = $product->variants->firstWhere('is_default', true) ?? $product->variants->first();
        }

        if (! $variant) {
            return;
        }

        // Calculate price
        $price = $this->calculatePrice($product, $variant);

        // Add to cart
        $cart = $this->getCart();
        $cart->addItem($variant->id, $qty, $price);

        $this->dispatch('cart:updated');
    }

    public function updateQuantity(int $itemId, int $quantity): void
    {
        $cart = $this->getCart();
        $item = $cart->items()->find($itemId);

        if (! $item) {
            return;
        }

        if ($quantity <= 0) {
            $item->delete();
        } else {
            $item->update(['quantity' => $quantity]);
        }

        $this->dispatch('cart:updated');
    }

    public function incrementQuantity(int $itemId): void
    {
        $cart = $this->getCart();
        $item = $cart->items()->find($itemId);

        if ($item) {
            $item->increment('quantity');
            $this->dispatch('cart:updated');
        }
    }

    public function decrementQuantity(int $itemId): void
    {
        $cart = $this->getCart();
        $item = $cart->items()->find($itemId);

        if ($item) {
            if ($item->quantity <= 1) {
                $item->delete();
            } else {
                $item->decrement('quantity');
            }
            $this->dispatch('cart:updated');
        }
    }

    public function removeItem(int $itemId): void
    {
        $cart = $this->getCart();
        $cart->items()->where('id', $itemId)->delete();

        $this->dispatch('cart:updated');
    }

    public function clearCart(): void
    {
        $this->getCart()->clear();

        $this->dispatch('cart:updated');
    }

    protected function calculatePrice(Product $product, ProductVariant $variant): int
    {
        $client = auth('client')->user();
        $priceType = 'retail';

        if ($client) {
            $t = mb_strtolower((string) ($client->client_type ?? ''));
            $hasCompanyName = trim((string) ($client->company ?? '')) !== '';
            $isCompany = $hasCompanyName || str_contains($t, 'company') || str_contains($t, 'شركة');
            $priceType = $isCompany ? 'wholesale' : 'retail';
        }

        $discountPercent = (int) ($product->discount_percent ?? 0);

        $row = ProductPrice::query()
            ->where('product_id', $product->id)
            ->where('product_variant_id', $variant->id)
            ->first();

        $current = 0;

        if ($row) {
            $current = $priceType === 'wholesale'
                ? (int) ($row->wholesale_price ?? 0)
                : (int) ($row->retail_price ?? 0);
        }

        if ($current <= 0) {
            $cost = (int) ($variant->cost ?? 0);
            $margin = 0.0;

            if ($row) {
                $margin = $priceType === 'wholesale'
                    ? (float) ($row->wholesale_margin ?? 10)
                    : (float) ($row->retail_margin ?? 20);
            } else {
                $margin = $priceType === 'wholesale' ? 10.0 : 20.0;
            }

            $current = (int) round($cost * (1 + $margin / 100));
        }

        if ($discountPercent > 0) {
            $current = (int) round(($current * (100 - $discountPercent)) / 100);
        }

        return $current;
    }

    public function formatPrice(int $value): string
    {
        return number_format($value, 0, '.', ',').' د.ع';
    }

    public function getAvailableCouponsProperty(): Collection
    {
        return Coupon::valid()
            ->orderBy('discount_value', 'desc')
            ->take(3)
            ->get()
            ->map(function ($coupon) {
                return [
                    'code' => $coupon->code,
                    'name' => $coupon->name,
                    'description' => $coupon->type === 'percentage'
                        ? 'خصم '.$coupon->discount_value.'%'
                        : 'خصم '.number_format($coupon->discount_value).' د.ع',
                    'min_purchase' => $coupon->min_purchase_amount,
                ];
            });
    }

    public function render()
    {
        return view('livewire.store.cart-manager', [
            'items' => $this->items,
            'count' => $this->count,
            'subtotal' => $this->subtotal,
            'shipping' => $this->calculatedShipping,
            'total' => $this->total,
            'shippingProgress' => $this->shippingProgress,
            'remainingForFreeShipping' => $this->remainingForFreeShipping,
            'availableCoupons' => $this->availableCoupons,
        ]);
    }
}
