<?php

namespace App\Livewire\Products;

use App\Models\Product;
use App\Models\ProductPrice;
use Livewire\Component;

class ProductCard extends Component
{
    public int $productId;

    public function getProductProperty(): Product
    {
        return Product::query()
            ->with(['category', 'variants' => fn ($q) => $q->where('is_active', true)])
            ->findOrFail($this->productId);
    }

    public function getBadgeProperty(): ?string
    {
        // أولوية للـ badge الخاص بالمنتج
        if (! empty($this->product->badge_text)) {
            return $this->product->badge_text;
        }

        // عرض نسبة الخصم إذا كان هناك خصم
        $price = $this->price;

        if ($price['percent'] > 0) {
            return "خصم {$price['percent']}%";
        }

        return null;
    }

    public function getBadgeTypeProperty(): ?string
    {
        // إذا كان هناك خصم، نجعل النوع sale
        $price = $this->price;

        if ($price['percent'] > 0) {
            return 'sale';
        }

        if (! empty($this->product->badge_type)) {
            return (string) $this->product->badge_type;
        }

        if ($this->badge) {
            return 'hot';
        }

        return null;
    }

    public function getCategoryProperty(): string
    {
        return $this->product->category?->name ?? '—';
    }

    public function getRatingProperty(): ?string
    {
        $rating = $this->product->avg_rating ?? null;

        return $rating !== null ? number_format((float) $rating, 1, '.', '') : null;
    }

    public function getPriceTypeProperty(): string
    {
        $client = auth('client')->user();

        if (! $client) {
            return 'retail';
        }

        $t = mb_strtolower((string) ($client->client_type ?? ''));
        $hasCompanyName = trim((string) ($client->company ?? '')) !== '';

        $isCompany = $hasCompanyName || str_contains($t, 'company') || str_contains($t, 'شركة');

        return $isCompany ? 'wholesale' : 'retail';
    }

    public function getPriceProperty(): array
    {
        $variant = $this->getDefaultVariant();

        if (! $variant) {
            return ['current' => 0, 'old' => 0, 'percent' => 0];
        }

        $row = ProductPrice::query()
            ->where('product_id', $this->productId)
            ->where('product_variant_id', (int) $variant->id)
            ->first();

        $originalPrice = 0;
        $current = 0;

        if ($row) {
            $originalPrice = $this->priceType === 'wholesale'
                ? (int) ($row->wholesale_price ?? 0)
                : (int) ($row->retail_price ?? 0);
        }

        if ($originalPrice <= 0) {
            $cost = (int) ($variant->cost ?? 0);

            $margin = 0.0;

            if ($row) {
                $margin = $this->priceType === 'wholesale'
                    ? (float) ($row->wholesale_margin ?? 10)
                    : (float) ($row->retail_margin ?? 20);
            } else {
                $margin = $this->priceType === 'wholesale' ? 10.0 : 20.0;
            }

            $originalPrice = (int) round($cost * (1 + $margin / 100));
        }

        // التحقق من وجود عرض خاص
        $salePrice = 0;
        $discountPercent = 0;

        if ($row && $row->isOnSale()) {
            if ($row->sale_price > 0) {
                $salePrice = (int) $row->sale_price;
            } elseif ($row->discount_percentage > 0) {
                $salePrice = (int) round($originalPrice * (1 - $row->discount_percentage / 100));
            }

            if ($salePrice > 0 && $salePrice < $originalPrice) {
                $discountPercent = (int) round((($originalPrice - $salePrice) / $originalPrice) * 100);
                $current = $salePrice;
            } else {
                $current = $originalPrice;
            }
        } else {
            $current = $originalPrice;
        }

        return [
            'current' => (int) $current,
            'old' => $current < $originalPrice ? (int) $originalPrice : 0,
            'percent' => $discountPercent,
        ];
    }

    public function getMainImageProperty(): ?string
    {
        $p = $this->product;

        // Use Spatie Media Library attributes
        return $p->thumbnail_thumb_url ?? $p->thumbnail_url ?? null;
    }

    public function getSecondImageProperty(): ?string
    {
        $p = $this->product;

        // Get gallery images or fallback to thumbnail
        $gallery = $p->gallery_thumb_urls ?? $p->gallery_urls ?? [];

        return $gallery[0] ?? $this->mainImage;
    }

    public function getColorsPreviewProperty(): array
    {
        $p = $this->product;

        $colors = $p->colors ?? null;

        if (is_array($colors)) {
            return array_values(array_slice($colors, 0, 2));
        }

        if (is_string($colors) && trim($colors) !== '') {
            $arr = array_map('trim', explode(',', $colors));

            return array_values(array_slice(array_filter($arr), 0, 2));
        }

        return [];
    }

    private function getDefaultVariant()
    {
        $variants = $this->product->variants;

        return $variants->firstWhere('is_default', true) ?? $variants->first();
    }

    public function formatIQD(int $value): string
    {
        return number_format((int) $value, 0, '.', ',').' د.ع';
    }

    // Actions
    public function addToCart(): void
    {
        $variant = $this->getDefaultVariant();
        $this->dispatch('cart:add', id: $this->productId, qty: 1, variantId: $variant?->id);
        $this->dispatch('cart-open'); // Browser event for Alpine
    }

    public function toggleWishlist(): void
    {
        $this->dispatch('wishlist:toggle', productId: $this->productId);
        $this->dispatch('wishlist-open'); // Browser event for Alpine
    }

    public function toggleCompare(): void
    {
        $this->dispatch('compare:toggle', productId: $this->productId);
        $this->dispatch('compare-open'); // Browser event for Alpine
    }

    public function quickView(): void
    {
        $this->dispatch('quickview-open', productId: $this->productId);
    }

    public function render()
    {
        return view('livewire.products.product-card', [
            'product' => $this->product,
            'badge' => $this->badge,
            'badgeType' => $this->badgeType,
            'category' => $this->category,
            'rating' => $this->rating,
            'price' => $this->price,
            'priceType' => $this->priceType, // ✅ أضف هذا السطر
            'mainImg' => $this->mainImage,
            'secondImg' => $this->secondImage ?: $this->mainImage,
            'colorsPreview' => $this->colorsPreview,
        ]);
    }
}
