<?php

namespace App\Livewire;

use App\Models\ProductDesign;
use App\Models\ProductModel;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Title;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.master')]
#[Title('مكتبة التصاميم')]
class DesignLibrary extends Component
{
    use WithPagination;

    #[Url(as: 'q')]
    public string $search = '';

    #[Url(as: 'sort')]
    public string $sortBy = 'popular'; // popular | newest | featured

    #[Url(as: 'view')]
    public string $viewMode = 'grid'; // grid | list

    #[Url(as: 'cols')]
    public int $gridColumns = 3; // 2, 3

    public bool $showFilters = false;

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    public function updatedSortBy(): void
    {
        $this->resetPage();
    }

    public function setViewMode(string $mode): void
    {
        $this->viewMode = $mode;
    }

    public function setColumns(int $cols): void
    {
        $this->gridColumns = $cols;
    }

    public function clearFilters(): void
    {
        $this->reset(['search', 'sortBy']);
        $this->resetPage();
    }

    public function render()
    {
        // Product Models query (main content)
        $modelsQuery = ProductModel::query()
            ->active()
            ->with(['product'])
            ->withCount('designs');

        // Search
        if ($this->search) {
            $modelsQuery->where(function ($q) {
                $q->where('name', 'like', '%'.$this->search.'%')
                    ->orWhere('description', 'like', '%'.$this->search.'%')
                    ->orWhereHas('product', fn ($query) => $query->where('name', 'like', '%'.$this->search.'%'));
            });
        }

        // Sorting
        match ($this->sortBy) {
            'newest' => $modelsQuery->latest(),
            'popular' => $modelsQuery->orderByDesc('designs_count'),
            'featured' => $modelsQuery->where('is_featured', true)->latest(),
            default => $modelsQuery->orderByDesc('designs_count'),
        };

        $models = $modelsQuery->paginate(12);

        // Top 6 popular designs
        $popularDesigns = ProductDesign::query()
            ->public()
            ->popular()
            ->with(['productModel.product', 'client'])
            ->limit(6)
            ->get();

        return view('livewire.design-library', [
            'models' => $models,
            'popularDesigns' => $popularDesigns,
        ]);
    }
}
