<?php

namespace App\Livewire\Auth;

use App\Models\Client;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class RegisterForm extends Component
{
    public string $name = '';

    public string $phone = '';

    public string $client_type = 'personal'; // personal | company | designer

    public ?string $company = null;

    public string $password = '';

    public string $password_confirmation = '';

    public bool $terms = false;

    public bool $newsletter = false;

    private function normalizePhone(string $input): ?string
    {
        $p = preg_replace('/\D+/', '', $input) ?? '';
        if ($p === '') {
            return null;
        }

        if (preg_match('/^07\d{9}$/', $p)) {
            return '00964'.substr($p, 1);
        }

        if (preg_match('/^00964\d{9}$/', $p)) {
            return $p;
        }

        return null;
    }

    public function register(): void
    {
        $rules = [
            'name' => ['required', 'string', 'min:3'],
            'phone' => ['required'],
            'client_type' => ['required', 'in:personal,company,designer'],
            'password' => ['required', 'string', 'min:6', 'confirmed'],
            'terms' => ['accepted'],
        ];

        if ($this->client_type === 'company') {
            $rules['company'] = ['required', 'string', 'min:2'];
        }

        $this->validate($rules, [
            'name.required' => 'الاسم مطلوب',
            'name.min' => 'الاسم يجب أن يكون 3 أحرف على الأقل',
            'phone.required' => 'رقم الهاتف مطلوب',
            'password.required' => 'كلمة المرور مطلوبة',
            'password.min' => 'كلمة المرور يجب أن تكون 6 أحرف على الأقل',
            'password.confirmed' => 'تأكيد كلمة المرور غير متطابق',
            'terms.accepted' => 'يجب الموافقة على الشروط والأحكام',
            'company.required' => 'اسم الشركة مطلوب',
        ]);

        $normalized = $this->normalizePhone($this->phone);

        if (! $normalized) {
            $this->addError('phone', 'رقم الهاتف يجب أن يكون 11 رقم يبدأ بـ 07 أو 14 رقم يبدأ بـ 00964.');

            return;
        }

        // تأكد uniqueness على الرقم الموحّد
        if (Client::where('phone', $normalized)->exists()) {
            $this->addError('phone', 'رقم الهاتف مستخدم مسبقًا.');

            return;
        }

        $branchId = (int) (user_info('branch_id') ?? 1);

        $client = new Client;
        $client->branch_id = $branchId;
        $client->name = trim($this->name);
        $client->phone = $normalized;
        $client->client_type = $this->client_type;
        $client->company = $this->client_type === 'company' ? (string) trim((string) $this->company) : null;
        $client->password = $this->password;
        $client->is_active = true;
        $client->save();

        Auth::guard('client')->login($client);
        session()->regenerate();

        $this->redirect(route('home'), navigate: true);
    }

    public function render()
    {
        return view('livewire.auth.register-form')->layout('layouts.master');
    }
}
