<?php

namespace App\Http\Controllers;

use App\Models\Wishlist;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class WishlistController extends Controller
{
    /**
     * Get all wishlist items
     */
    public function index(): JsonResponse
    {
        $clientId = Auth::guard('client')->id();

        if (! $clientId) {
            return response()->json([
                'success' => false,
                'message' => 'يجب تسجيل الدخول',
            ], 401);
        }

        $wishlist = Wishlist::where('client_id', $clientId)
            ->with(['product.variants', 'variant'])
            ->get();

        $items = $wishlist->map(function ($item) {
            $variant = $item->variant ?? $item->product->defaultVariant ?? $item->product->variants->first();

            return [
                'id' => $item->id,
                'product_id' => $item->product_id,
                'product_name' => $item->product->name,
                'product_slug' => $item->product->slug,
                'variant_id' => $variant?->id,
                'variant_name' => $variant?->name,
                'price' => $variant?->retail_price ?? 0,
                'image' => $item->product->thumbnail_thumb_url,
                'in_stock' => $item->product->isInStock(),
            ];
        });

        return response()->json([
            'success' => true,
            'items' => $items,
            'count' => $items->count(),
        ]);
    }

    /**
     * Toggle product in wishlist
     */
    public function toggle(Request $request): JsonResponse
    {
        $clientId = Auth::guard('client')->id();

        if (! $clientId) {
            return response()->json([
                'success' => false,
                'message' => 'يجب تسجيل الدخول',
            ], 401);
        }

        $validated = $request->validate([
            'product_id' => 'required|integer|exists:products,id',
            'variant_id' => 'nullable|integer|exists:product_variants,id',
        ]);

        $wishlist = Wishlist::where('client_id', $clientId)
            ->where('product_id', $validated['product_id'])
            ->when(isset($validated['variant_id']), fn ($q) => $q->where('product_variant_id', $validated['variant_id']))
            ->first();

        if ($wishlist) {
            $wishlist->delete();

            return response()->json([
                'success' => true,
                'action' => 'removed',
                'message' => 'تم إزالة المنتج من المفضلة',
            ]);
        }

        Wishlist::create([
            'client_id' => $clientId,
            'product_id' => $validated['product_id'],
            'product_variant_id' => $validated['variant_id'] ?? null,
        ]);

        return response()->json([
            'success' => true,
            'action' => 'added',
            'message' => 'تمت إضافة المنتج للمفضلة',
        ]);
    }

    /**
     * Remove item from wishlist
     */
    public function remove(Request $request): JsonResponse
    {
        $clientId = Auth::guard('client')->id();

        if (! $clientId) {
            return response()->json([
                'success' => false,
                'message' => 'يجب تسجيل الدخول',
            ], 401);
        }

        $validated = $request->validate([
            'product_id' => 'required|integer',
        ]);

        Wishlist::where('client_id', $clientId)
            ->where('product_id', $validated['product_id'])
            ->delete();

        return response()->json([
            'success' => true,
            'message' => 'تم إزالة المنتج من المفضلة',
        ]);
    }

    /**
     * Clear wishlist
     */
    public function clear(): JsonResponse
    {
        $clientId = Auth::guard('client')->id();

        if (! $clientId) {
            return response()->json([
                'success' => false,
                'message' => 'يجب تسجيل الدخول',
            ], 401);
        }

        Wishlist::where('client_id', $clientId)->delete();

        return response()->json([
            'success' => true,
            'message' => 'تم تفريغ المفضلة',
        ]);
    }
}
